//============================================================================
// Name        : personalized-pagerank.cpp
// Author      : 
// Version     :
// Copyright   : Your copyright notice
// Description : Hello World in C, Ansi-style
//============================================================================


#include <stdio.h>
#include <stdlib.h>
#include <algorithm>
#include <vector>
#include "Snap.h"
#include "pagerank_algorithms.h"
#include "graph_names.h"





void computeAccuracyMeasurements(const std::vector<double>& trueValues, const std::vector<double>& estimates, double threshold, double approximationRatio, double ratioOfRelevance, std::vector<double> *measurementVector) {
	double maxError = 0.0;
	double sumError = 0.0;
	double maxRelativeError = 0.0;
	double sumRelativeError = 0.0;
	int nRelevantPairs = 0; // "relevant" means threshold / ratioOfRelevance <= true PPR <= threshold * ratioOfRelevance
	double actualRatio = 1.0;

	int truePositiveCount = 0; // "Positive" means pi(start, target) > threshold
	int falsePositiveCount = 0;
	int trueNegativeCount = 0;
	int falseNegativeCount = 0;

	for (unsigned i = 0; i < trueValues.size(); i++) {
		double truePPR = trueValues[i];
		double estimate = estimates[i];
		bool inBorderRegion = threshold / approximationRatio < truePPR && truePPR < threshold * approximationRatio;
		//printf( "%s in border region ", inBorderRegion ? "   " : "not");
		if (truePPR >= threshold && estimate >= threshold) {
			//printf("True positive :)\n");
			truePositiveCount++;
		} else if (truePPR < threshold && estimate < threshold) {
			//printf("True negative :)\n");
			trueNegativeCount++;
		} else if (truePPR >= threshold && estimate < threshold) {
			//printf("False Negative\n");
			falseNegativeCount++;
			if (!inBorderRegion) {
				printf("Guarantee violated!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
			}
		} else {
			//printf("False Positive\n");
			falsePositiveCount++;
			if (!inBorderRegion) {
				printf("Guarantee violated!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
			}
		}
		if (threshold / ratioOfRelevance < truePPR && truePPR < threshold * ratioOfRelevance) {
			nRelevantPairs++;
			double error = fabs(estimate - truePPR);
			maxError = fmax(maxError, error);
			sumError += error;
			maxRelativeError = fmax(maxRelativeError, error / truePPR);
			sumRelativeError += error / truePPR;
		}
		if (estimate < threshold) {
			// If we classified true PPR as small, our approximation ratio is as bad as the actual amount truePPR is large
			actualRatio = fmax(actualRatio, truePPR / threshold);
		} else if (estimate >= threshold) {
			// If we classified true PPR as large, our approximation ratio is as bad as the actual amount truePPR is small
			actualRatio = fmax(actualRatio, threshold / truePPR );
		}
	}

	printf("Threshold=%e, ratioOfRelevance=%f\n", threshold, ratioOfRelevance);

	printf("Average Additive Error: %e\n", sumError / nRelevantPairs);
	printf("Max Additive Error: %e\n", maxError);
	printf("Average Relative Error: %f\n", sumRelativeError / nRelevantPairs);
	printf("Max Relative Error: %f\n", maxRelativeError);
	printf("Empirical Approximation Ratio: %f\n", actualRatio);

	printf("\n");
	printf("Positives (true/total): %d/%d\n", truePositiveCount, truePositiveCount + falsePositiveCount);
	printf("Negatives (true/total): %d/%d\n", trueNegativeCount, trueNegativeCount + falseNegativeCount);
	double f1Score = 2.0 * truePositiveCount / (2.0 * truePositiveCount + falseNegativeCount + falsePositiveCount);
	printf("F1 score: %f\n", f1Score);
	/*
	Average Additive Error*:
	Max Additive Error*:
	Average Relative Error*:
	Max Relative Error*:
	Number of Relevant Pairs
	Empirical Approximation Ratio:
	True Positive
	False Positive
	True Negative
	False Negative
	F1
	*/
	measurementVector->push_back(sumError / nRelevantPairs);
	measurementVector->push_back(maxError);
	measurementVector->push_back( sumRelativeError / nRelevantPairs);
	measurementVector->push_back( maxRelativeError);
	measurementVector->push_back(nRelevantPairs);
	measurementVector->push_back(actualRatio);
	measurementVector->push_back(truePositiveCount);
	measurementVector->push_back(falsePositiveCount);
	measurementVector->push_back(trueNegativeCount);
	measurementVector->push_back(falseNegativeCount);
	measurementVector->push_back(f1Score);

	printf("truePPRs = ");
	printDoublesForPython(trueValues);
	printf("\nestimates = ");
	printDoublesForPython(estimates);
	printf("\n\n");
}

void accuracyToTargets(const PNGraph& graph, double teleportProb, bool uniformSampling, double threshold, double approximationRatio, double failureProbability, TRnd& rnd,
		std::vector<double> *measurementVector, bool includeFastPPR, bool includeMonteCarlo, bool includeDynamicProgramming) {
	bool verbose = true;
	int nTargets = 10;
	unsigned nStartsPerTarget = 100;
	double ratioOfRelevance = 4.0; // We will compute our error only if threshold / ratioOfRelevance < true ppr < threshold * ratioOfRelevance

	std::vector<double> truePPRs;
	std::vector<double> estimatedPPRsWithFrontier;
	std::vector<double> estimatedPPRsWithoutFrontier;
	std::vector<double> estimatedPPRsMonteCarlo;
	std::vector<double> estimatedPPRsDynamicProgramming;

	for (int trial = 0; trial < nTargets; trial++) {
		TNGraph::TNodeI target = uniformSampling ? graph->GetRndNI(rnd) : samplePageRank(graph, teleportProb, rnd);
		double startTime = wallClockTime();
		printf("Computing true contributions for target %d/%d ...\n", trial, nTargets);
		THash<TInt, TFlt> trueContributions = computeContributions(graph,
				target, teleportProb, threshold / 100.0, NULL);
		printf("Computing true contributions took %g s\n", wallClockTime() - startTime);
		printf("target node: %d\n", target.GetId());

		THash<TInt, TFlt> dpEstimates;
		if (includeDynamicProgramming) {
			dpEstimates = computeContributions(graph,
						target, teleportProb, threshold / 2.0, NULL);
		}
		std::vector<int> relevantNegativeExamples;
		std::vector<int> relevantPositiveExamples;
		for (TNGraph::TNodeI u = graph->BegNI(); u != graph->EndNI(); u++) {
			int uId = u.GetId();
			if (trueContributions.IsKey(uId)) {
				double truePPR = trueContributions.GetDat(uId);
				if (threshold / ratioOfRelevance < truePPR
						&& truePPR < threshold) {
					relevantNegativeExamples.push_back(uId);
				} else if (threshold < truePPR
						&& truePPR < threshold * ratioOfRelevance) {
					relevantPositiveExamples.push_back(uId);
				}
			}
		}
		printf("# relevant positive: %d\n", (int) relevantPositiveExamples.size());
		printf("# relevant negative: %d\n", (int) relevantNegativeExamples.size());


		for (unsigned i = 0; i < nStartsPerTarget; i++) {
			TNGraph::TNodeI start;
			if ( i < nStartsPerTarget / 2 && relevantPositiveExamples.size() >= nStartsPerTarget / 2) {
				start = graph->GetNI(relevantPositiveExamples.at(rnd.GetUniDevInt(relevantPositiveExamples.size())));
			} else if ( i >= nStartsPerTarget / 2 && relevantNegativeExamples.size() >= nStartsPerTarget / 2) {
				start = graph->GetNI(relevantNegativeExamples.at(rnd.GetUniDevInt(relevantNegativeExamples.size())));
			} else {
				continue;
			}
			double truePPR =
					trueContributions.IsKey(start.GetId()) ?
							(double) trueContributions.GetDat(start.GetId()) :
							0.0;
			truePPRs.push_back(truePPR);
			double estimateWithFrontier = 0.0;
			double estimateWithoutFrontier = 0.0;
			if (includeFastPPR) {
				estimateWithFrontier = fastPPR(graph, start, target,
						teleportProb,

						threshold, approximationRatio, failureProbability, rnd,
						verbose, true);
				estimateWithoutFrontier = fastPPR(graph, start, target,
						teleportProb, threshold, approximationRatio,
						failureProbability, rnd, verbose, false);

			}
			double estimateMonteCarlo = 0.0;
			if (includeMonteCarlo) {
				long long int nWalks = 1.0 / threshold;
				printf("nWalks: %lld\n", nWalks);
				THash<TInt, TFlt> estimates = monteCarloPPR(graph, start,  teleportProb,
												nWalks, rnd, NULL, NULL);
				estimateMonteCarlo = estimates.IsKey(target.GetId()) ? (double) estimates.GetDat(target.GetId()) : 0.0;
			}
			double dpEstimate = 0.0;
			if (includeDynamicProgramming) {
				dpEstimate = dpEstimates.IsKey(start.GetId()) ?
						(double) dpEstimates.GetDat(start.GetId()) :
						0.0;
			}
			estimatedPPRsWithFrontier.push_back(estimateWithFrontier);
			estimatedPPRsWithoutFrontier.push_back(estimateWithoutFrontier);
			estimatedPPRsMonteCarlo.push_back(estimateMonteCarlo);
			estimatedPPRsDynamicProgramming.push_back(dpEstimate);
			printf("%10d  %10e  %10e  %10e  %10e  %10e\n", start.GetId(), truePPR, estimateWithFrontier, estimateWithoutFrontier, estimateMonteCarlo, dpEstimate);
		}
	}

	//int measurementOffsetWithoutFrontier = measurementVector->size() / 2;

	printf("nEdges = %d\n", graph->GetEdges());

	printf("teleportProb = %f\n", teleportProb);
	printf("uniformSampling = %s\n", uniformSampling? "true" : "false");
	if (includeFastPPR) {
		printf("With Frontier:\n");
		computeAccuracyMeasurements(truePPRs, estimatedPPRsWithFrontier, threshold, approximationRatio, ratioOfRelevance,  measurementVector);
		printf("Without Frontier:\n");
		computeAccuracyMeasurements(truePPRs, estimatedPPRsWithoutFrontier, threshold, approximationRatio, ratioOfRelevance, measurementVector);
	}
	if (includeMonteCarlo) {
		printf("Using Monte Carlo:\n");
		computeAccuracyMeasurements(truePPRs, estimatedPPRsMonteCarlo, threshold, approximationRatio, ratioOfRelevance, measurementVector);
	}
	if (includeDynamicProgramming) {
			printf("Using DP:\n");
			computeAccuracyMeasurements(truePPRs, estimatedPPRsDynamicProgramming, threshold, approximationRatio, ratioOfRelevance, measurementVector);
	}
}

std::vector<int> topIndices(std::vector<double> values, PNGraph graph) {
	std::vector< std::pair<double, int> > valuePairs(graph->GetNodes(), std::make_pair(0.0, 0));
	std::vector<int> result(valuePairs.size(), 0);
	int i = 0;
	for (TNGraph::TNodeI u = graph->BegNI(); u != graph->EndNI(); u++) {
		int uId = u.GetId();
		valuePairs[i] = std::make_pair(values[uId], uId);
		i++;
	}
	std::sort(valuePairs.begin(), valuePairs.end());
	for (unsigned i = 0; i < valuePairs.size(); i++) {
		result.at(i) = valuePairs.at(valuePairs.size() - i - 1).second;
	}
	return result;
}


enum ppr_method_type {FAST_PPR = 0, FAST_PPR_PR_BALANCED = 1, PUSH_FROM_TARGET = 2, MONTE_CARLO_PPR = 3};
int nTrialsPerMethod[] = {1000, 1000, 20, 5};
enum sampling_method_type {SAMPLE_UNIFORM = 0, SAMPLE_PAGERANK = 1, SAMPLE_WORST_CASE = 2, SAMPLE_REPRESENTITIVE_PAGERANK = 3};


void timePPR(ppr_method_type pprMethod, const PNGraph& graph, double teleportProb, sampling_method_type samplingMethod, double threshold,  double approximationRatio, double failureProbability,  TRnd& rnd,
		double *meanExecutionTime, double *meanStepCount) {
	bool verbose = true;
	bool expandInNeighbors = true;
	TVec<double> runTimes;
	TVec<double> forwardTimes;
	TVec<double> backwardTimes;
	TVec<double> stepCounts;

	TVec<double> targetPageRanks;

	int nTrials = nTrialsPerMethod[pprMethod];
	if (samplingMethod == SAMPLE_WORST_CASE || samplingMethod == SAMPLE_REPRESENTITIVE_PAGERANK)
		nTrials = 100;

	double totalForwardTime = 0.0;
	double totalBackwardTime = 0.0;
	printf("Timing PPR method #: %d\n", (int) pprMethod);
	std::vector<double> globalPR;
	std::vector<int> topPageRankNodes;
	if (pprMethod == FAST_PPR_PR_BALANCED || samplingMethod == SAMPLE_WORST_CASE || samplingMethod == SAMPLE_REPRESENTITIVE_PAGERANK) {
		printf("Computing global PageRank...\n");
		globalPR = globalPageRank(graph, teleportProb, 1.0e-9);
		printf("We computed global PageRank!\n");
		if (samplingMethod == SAMPLE_WORST_CASE || samplingMethod == SAMPLE_REPRESENTITIVE_PAGERANK) {
			topPageRankNodes = topIndices(globalPR, graph);
		}
	}
	std::vector<double> emptyVector;

	double edgeCount = countEdges(graph); // Needed for FastPPR

	for (int i = 0; i < nTrials; i++) {
		double startTime = wallClockTime();
		double forwardTime = 0.0;
		double backwardTime = 0.0;
		long long stepCount = 0;
		TNGraph::TNodeI start = graph->GetRndNI(rnd);
		TNGraph::TNodeI target;
		if (samplingMethod == SAMPLE_UNIFORM)
			target = graph->GetRndNI(rnd);
		else if (samplingMethod == SAMPLE_PAGERANK)
			target = samplePageRank(graph, teleportProb, rnd);
		else if (samplingMethod == SAMPLE_WORST_CASE) {
			int targetId = topPageRankNodes[i];
			target = graph->GetNI(targetId);

		} else {//SAMPLE_REPRESENTITIVE_PAGERANK
			int targetId = topPageRankNodes[((long long)topPageRankNodes.size()) * i / nTrials];
			target = graph->GetNI(targetId);
		}
		if (globalPR.size() > 0) {
			printf("Node %d has global pi %g\n", target.GetId(), globalPR[target.GetId()]);
			targetPageRanks.Add(globalPR[target.GetId()]);
		}
		double estimate = -1.0;
		if (pprMethod == FAST_PPR || pprMethod == FAST_PPR_PR_BALANCED) {
			estimate = fastPPR(graph, start, target, teleportProb,
					           threshold, approximationRatio, failureProbability, rnd, verbose, expandInNeighbors,
					           pprMethod == FAST_PPR_PR_BALANCED ? globalPR : emptyVector,
					           edgeCount, &forwardTime, &backwardTime, &stepCount);
		} else if (pprMethod == PUSH_FROM_TARGET) {
			THash<TInt, TFlt> estimates = computeContributions(graph, target, teleportProb, threshold / 2.0, &stepCount);
			estimate = estimates.IsKey(start.GetId()) ? (double) estimates.GetDat(start.GetId()) : 0.0;
		} else if (pprMethod == MONTE_CARLO_PPR) {
			//long long int nWalks = 8.0 / threshold * log(1.0 / failureProbability);
			long long int nWalks = 35.0 / threshold;
			printf("nWalks: %lld\n", nWalks);
			THash<TInt, TFlt> estimates= monteCarloPPR(graph, start,  teleportProb,
								nWalks, rnd, NULL, &stepCount);
			estimate = estimates.IsKey(start.GetId()) ? (double) estimates.GetDat(start.GetId()) : 0.0;
		}

		if (estimate == -1.2345)
			printf("Hey optimizing compiler, we looked at estimate, just to make sure you don't optimize it away.\n");
		runTimes.Add(wallClockTime() - startTime);
		stepCounts.Add(stepCount);
		if (pprMethod == FAST_PPR || pprMethod == FAST_PPR_PR_BALANCED) {
			printf("forwardTime + backwardTime = %g + %g\n", forwardTime, backwardTime);
			totalForwardTime += forwardTime;
			totalBackwardTime += backwardTime;
			forwardTimes.Add(forwardTime);
			backwardTimes.Add(backwardTime);
		} else {
			printf("runTime: %g\n", wallClockTime() - startTime);
		}
	}
	printf("pprMethod = %d\n", pprMethod);

	printf("nEdges = %d\n", graph->GetEdges());
	printf("nTrials = %d = %d\n", nTrials, runTimes.Len());
	if (pprMethod == FAST_PPR || pprMethod == FAST_PPR_PR_BALANCED) {
			printf("mean = forwardMean + backwardMean = %g + %g\n", totalForwardTime / nTrials, totalBackwardTime / nTrials);
			printf("forwardTimes = ");
			printDoubles(forwardTimes);
			printf("backwardTimes = ");
			printDoubles(backwardTimes);
	}
	printf("pageRanks = ");
	printDoubles(targetPageRanks);
	printf("runTimes = ");
	printDoubles(runTimes);
	printf("meanRunTime = %f; 90%% interval: (%f, %f)\n", mean(runTimes), percentile(runTimes, 5), percentile(runTimes, 95));
	printf("stepCounts = ");
	printDoubles(stepCounts);
	printf("meanStepCount = %f; 90%% interval: (%f, %f)\n", mean(stepCounts), percentile(stepCounts, 5), percentile(stepCounts, 95));
	*meanExecutionTime = mean(runTimes);
	*meanStepCount = mean(stepCounts);


}
/*
void frontierSizeWorstCase(PNGraph g, double teleportProb, double threshold, TRnd& rnd) {

	std::vector<double> pi = globalPageRank(g, teleportProb, 1.0e-9);
	std::vector< std::pair<double, int> > piPairs(pi.size(), std::make_pair(0.0, 0));
	for (TNGraph::TNodeI u = g->BegNI(); u != g->EndNI(); u++) {
		int uId = u.GetId();
		piPairs[uId] = std::make_pair(pi[uId], uId);
	}
	std::sort(piPairs.begin(), piPairs.end());
	for (unsigned i = piPairs.size() - 1; i >= piPairs.size() - 100; i--) {
		int uId = piPairs[i].second;
		printf("Node %d has global pi %g\n", uId, pi[uId]);
		// We don't care about forward direction, so failureProbability can be large
		double failureProbability = 1.0;
		fastPPR(g, g->BegNI(), g->GetNI(uId),
				teleportProb, threshold, 2.0, failureProbability, rnd, true, true);

	}
}

void runtimeWorstCase(PNGraph g, double teleportProb, double threshold, TRnd& rnd) {

	std::vector<double> pi = globalPageRank(g, teleportProb, 1.0e-9);
	double edgeCount = countEdges(g);
	std::vector< std::pair<double, int> > piPairs(pi.size(), std::make_pair(0.0, 0));
	for (TNGraph::TNodeI u = g->BegNI(); u != g->EndNI(); u++) {
		int uId = u.GetId();
		piPairs[uId] = std::make_pair(pi[uId], uId);
	}
	std::sort(piPairs.begin(), piPairs.end());
	double forwardTime = 0.0;
	double reverseTime = 0.0;
	int nTop = 10;
	for (unsigned i = piPairs.size() - 1; i >= piPairs.size() - nTop; i--) {
		int uId = piPairs[i].second;
		printf("Node %d has global pi %g\n", uId, pi[uId]);
		double failureProbability = 1.0e-4;
		double forwardTimeForU = 0.0;
		double reverseTimeForU = 0.0;
		TNGraph::TNodeI start = g->GetRndNI(rnd);
		fastPPR(g, start, g->GetNI(uId),
				teleportProb, threshold, 2.0, failureProbability, rnd, true, true, pi, edgeCount, &forwardTimeForU, &reverseTimeForU);
		printf("Computation took time forward + reverse = %f = %f + %f\n", forwardTimeForU + reverseTimeForU, forwardTimeForU, reverseTimeForU);
		forwardTime += forwardTimeForU;
		reverseTime += reverseTimeForU;
	}
	printf("mean time forward + reverse = %f = %f + %f\n", (forwardTime + reverseTime) / nTop, forwardTime / nTop, reverseTime / nTop);
}
*/

PNGraph loadGraph(TStr graphFilename) {
	PNGraph graph;
	if(graphFilename.GetSubStr(graphFilename.Len() - 5) != ".snap") {
		printf("Converting %s to snap format...\n", graphFilename.CStr());
		graph = TSnap::LoadEdgeList<PNGraph>(graphFilename, false);
		TFOut fOut(graphFilename + ".snap");
		graph->Save(fOut);
	} else {
		printf("Loading Graph...\n");
		TFIn FIn(graphFilename);
		graph = TNGraph::Load(FIn);
	}
	printf("# nodes: %d\n", graph->GetNodes());
	printf("# edges: %g\n", countEdges(graph));

	return graph;
}

void undirectGraph(TStr graphFilename) {
	printf("graph: %s\n", graphFilename.CStr());
	PNGraph graph = loadGraph(graphFilename);
	PNGraph undirectedGraph = TNGraph::New();
	//undirectedGraph->Reserve(graph->GetNodes(), 2 * graph->GetEdges());
	for (TNGraph::TEdgeI EI = graph->BegEI(); EI < graph->EndEI(); EI++) {
		// In snap, why doesn't AddEdge automatically add the nodes?
		// And why does AddNode seg-fault if the node already exists?
		if (! undirectedGraph->IsNode(EI.GetSrcNId()))
			undirectedGraph->AddNode(EI.GetSrcNId());
		if (! undirectedGraph->IsNode(EI.GetDstNId()))
			undirectedGraph->AddNode(EI.GetDstNId());
		undirectedGraph->AddEdge(EI.GetSrcNId(), EI.GetDstNId());
		undirectedGraph->AddEdge(EI.GetDstNId(), EI.GetSrcNId());
		//printf("Adding edge %d, %d\n", EI.GetSrcNId(), EI.GetDstNId());
		//printf("Adding edge %d, %d\n", EI.GetDstNId(), EI.GetSrcNId());

	}
	printf("undirectedGraph->GetEdges() = %d\n", undirectedGraph->GetEdges());
	TFOut fOut(graphFilename + ".undirected.snap");
	undirectedGraph->Save(fOut);
}


int nPPRAlgorithms = 4;

void runtimeExperiment(double teleportProb, sampling_method_type samplingMethod) {

	std::vector< std::vector<double> > runTimes(nPPRAlgorithms, std::vector<double>(nGraphNames, 0.0));
	std::vector< std::vector<double> > stepCounts(nPPRAlgorithms, std::vector<double>(nGraphNames, 0.0));
	TRnd rnd(0);// Seed==0 means seed from clock

	for (int graphI = 0; graphI < nGraphNames; graphI++) {
		if (samplingMethod == SAMPLE_REPRESENTITIVE_PAGERANK && graphI != nGraphNames - 2)
			continue;
		printf("graph: %s\n", graphNames[graphI]);
		PNGraph graph = loadGraph(graphNames[graphI]);
		double threshold = 4.0 / graph->GetNodes();
		double approximationRatio = 1.5;
		 double failureProbability = 1.0e-6;

		int nAlgs = (samplingMethod != SAMPLE_WORST_CASE && samplingMethod != SAMPLE_REPRESENTITIVE_PAGERANK ? nPPRAlgorithms : 3);
		//for (int algorithmI =  MONTE_CARLO_PPR; algorithmI <=  MONTE_CARLO_PPR; algorithmI++) {
		for (int algorithmI =  0; algorithmI <  nAlgs; algorithmI++) {
			if (samplingMethod == SAMPLE_REPRESENTITIVE_PAGERANK && algorithmI != 1)
						continue;

						timePPR((ppr_method_type) algorithmI, graph, teleportProb, samplingMethod, threshold,  approximationRatio, failureProbability, rnd,
					&(runTimes[algorithmI][graphI]), &(stepCounts[algorithmI][graphI]));
		}



		printf("\nRuntimes = ");
		printMatrixForPython(runTimes);

		printf("\nStepCounts = ");
		printMatrixForPython(stepCounts);

		printf("\nRuntimes:\n");
		printMatrixTSV(runTimes);

		printf("\nStepCounts:\n");
		printMatrixTSV(stepCounts);
	}
}

void frontierExperiment(double teleportProb, bool uniformSampling) {
	std::vector< std::vector<double> > measurementsTranspose(nGraphNames, std::vector<double>());
	TRnd rnd(0);// Seed==0 means seed from clock

	for (int graphI = 0; graphI < nGraphNames; graphI++) {
		printf("graph: %s\n", graphNames[graphI]);
		PNGraph graph = loadGraph(graphNames[graphI]);
		double threshold = 4.0 / graph->GetNodes();
		double approximationRatio = 1.5;
		double failureProbability = 1.0e-6;

		accuracyToTargets(graph, teleportProb, uniformSampling, threshold, approximationRatio, failureProbability,  rnd,
				&(measurementsTranspose[graphI]), true, false, true);

		std::vector< std::vector<double> > measurements = transposeMatrix(measurementsTranspose);
		printf("\nMeasurements = ");
		printMatrixForPython(measurements);

		printf("\nMeasurements: \n");
		printMatrixTSV(measurements);
		graph->Clr();
	}
}


int main(int argc, char* argv[]) {
	setbuf(stdout, NULL);
	// PNGraph G = PNGraph::TObj::New();
	// char * graphFilename = "soc-pokec-relationships.txt";
	//char * graphFilename = "ca-GrQc.txt.gz";
	//char * graphFilename = "test_graph.txt";
	//char * graphFilename = "eht_experiment/com-orkut.ungraph.txt";
	if (strcmp(argv[1], "runtime") == 0) {
		if (argc < 4) {
					printf("Usage: %s runtime <teleport prob> <uniform, pagerank, or worse_case>\n", argv[0]);
					exit(0);
				}
		double teleportProb = atof(argv[2]);
		printf("teleportProb = %f\n", teleportProb);
		sampling_method_type samplingMethod;
		if (strcmp(argv[3], "uniform") == 0) {
			samplingMethod = SAMPLE_UNIFORM;
		} else if (strcmp(argv[3], "pagerank") == 0) {
			samplingMethod = SAMPLE_PAGERANK;
		} else if (strcmp(argv[3], "worst_case") == 0) {
			samplingMethod = SAMPLE_WORST_CASE;
		} else if (strcmp(argv[3], "representitive") == 0) {
			samplingMethod = SAMPLE_REPRESENTITIVE_PAGERANK;
		} else {
			printf("Usage: %s runtime <teleport prob> <uniform, pagerank, or worse_case>\n", argv[0]);
			exit(0);
		}
		printf("samplingMethod = %d\n", (int) samplingMethod);
		runtimeExperiment(teleportProb, samplingMethod);
	} else if (strcmp(argv[1], "frontier") == 0) {
		if (argc < 4) {
					printf("Usage: %s frontier <teleport prob> <uniform or pagerank>\n", argv[0]);
					exit(0);
				}
		double teleportProb = atof(argv[2]);
		printf("teleportProb = %f\n", teleportProb);
		bool uniformSampling = (strcmp(argv[3], "uniform") == 0);
		printf("uniformSampling = %s\n", uniformSampling? "true" : "false");
		frontierExperiment(teleportProb, uniformSampling);
	} else if (strcmp(argv[1], "undirect") == 0) {
		if (argc < 3) {
			printf("Usage: %s undirect <graph filename>\n", argv[0]);
			exit(0);
		}
		char* graphFilename = argv[2];
		undirectGraph(graphFilename);
	} else {

/*

		if (argc < 2) {
			printf("Usage: %s <graph filename>\n", argv[0]);
			exit(0);
		}
		char* graphFilename = argv[1];
		printf("graph: %s\n", graphFilename);
		PNGraph graph = loadGraph(graphFilename);

		TRnd rnd(0);// Seed==0 means seed from clock
		double teleportProb = 0.15;

		double threshold = 1.0 / graph->GetNodes(); //1.0e-6;
		printf("threshold delta=%e\n", threshold);
		double approximationRatio = 2.0;
		double failureProbability = 1.0e-3;

		if (false) {
			int methodNumber =  FAST_PPR;//atoi(argv[2]);
			//int nTrials = 400;
			double executionTime = 0.0;
			double stepCount = 0;
			timePPR((ppr_method_type) methodNumber, graph, teleportProb, threshold, approximationRatio, failureProbability, rnd, &executionTime, &stepCount);
		} else if (false) {
			bool expandInNeighbors = true;//(argc <= 2);
			printf("use full frontier: %s\n", expandInNeighbors ? "yes" : "no");
			accuracyToTargets(graph, teleportProb, threshold, approximationRatio, failureProbability, rnd, expandInNeighbors);
		} else {
			runtimeWorstCase(graph, teleportProb, threshold, rnd);
			//frontierSizeWorstCase(graph, teleportProb, threshold, rnd);
		}

		//
	//	for (TNGraph::TNodeI i = graph->BegNI(); i < graph->EndNI(); i++) {
	//		printf("%6d: %6d  %6d\n", i.GetId(), i.GetInDeg(), i.GetOutDeg());
	//	}

		*/
	}
	return EXIT_SUCCESS;
}
