# This network was made by combining the following two tensorflow examples:
#	https://github.com/tensorflow/tensorflow/blob/r0.8/tensorflow/models/image/alexnet/alexnet_benchmark.py
#   https://github.com/tensorflow/tensorflow/blob/r0.8/tensorflow/examples/tutorials/mnist/mnist.py
#	https://tensorflow.googlesource.com/tensorflow/+/master/tensorflow/models/image/mnist/convolutional.py

# Copyright 2015 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================


from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from datetime import datetime
import math
import time

from six.moves import xrange  # pylint: disable=redefined-builtin
import tensorflow as tf


FLAGS = tf.app.flags.FLAGS

tf.app.flags.DEFINE_integer('batch_size', 256,
                            """Batch size.""")
tf.app.flags.DEFINE_integer('num_batches', 50,
                            """Number of batches to run.""")


def print_activations(t):
  #print(t.op.name, ' ', t.get_shape().as_list())
  return


def inference(images):
  """Build the AlexNet model.

  Args:
    images: Images Tensor

  Returns:
    softmax: output probabilities from softmax
    parameters: a list of Tensors corresponding to the weights and biases of the
        AlexNet model.
  """
  parameters = []
  # conv1, relu
  with tf.name_scope('conv1') as scope:
    kernel = tf.Variable(tf.truncated_normal([11, 11, 3, 96], # [filter_height, filter_width, in_channels, out_channels]
                                             dtype=tf.float32,
                                             stddev=1e-1), name='weights')
    # https://www.tensorflow.org/versions/r0.8/api_docs/python/nn.html#conv2d
    conv = tf.nn.conv2d(images, kernel, [1, 4, 4, 1], # [batch, in_height, in_width, in_channels]
                        padding='VALID')               # conv1 has padding of 0
    biases = tf.Variable(tf.constant(0.0, shape=[96], dtype=tf.float32),
                         trainable=True, name='biases')
    bias = tf.nn.bias_add(conv, biases)
    conv1 = tf.nn.relu(bias, name=scope)
    print_activations(conv1)
    parameters += [kernel, biases]

  # pool1
  pool1 = tf.nn.max_pool(conv1,
                         ksize=[1, 3, 3, 1],
                         strides=[1, 2, 2, 1],
                         padding='VALID',
                         name='pool1')
  print_activations(pool1)

  # lrn1
  lrn1 = tf.nn.local_response_normalization(pool1,
                                            depth_radius=2, # local_size = 5
                                            alpha=0.0001,
                                            beta=0.75,
                                            bias=1.0)

  # conv2, relu
  with tf.name_scope('conv2') as scope:
    kernel = tf.Variable(tf.truncated_normal([5, 5, 96, 256], dtype=tf.float32,
                                             stddev=1e-1), name='weights')
    conv = tf.nn.conv2d(lrn1, kernel, [1, 1, 1, 1], padding='SAME') # conv2 has padding of 2
    biases = tf.Variable(tf.constant(0.0, shape=[256], dtype=tf.float32),
                         trainable=True, name='biases')
    bias = tf.nn.bias_add(conv, biases)
    conv2 = tf.nn.relu(bias, name=scope)
    parameters += [kernel, biases]
  print_activations(conv2)

  # pool2
  pool2 = tf.nn.max_pool(conv2,
                         ksize=[1, 3, 3, 1],
                         strides=[1, 2, 2, 1],
                         padding='VALID',
                         name='pool2')
  print_activations(pool2)

  # lrn2
  lrn2 = tf.nn.local_response_normalization(pool2,
                                            depth_radius=2,
                                            alpha=0.0001,
                                            beta=0.75,
                                            bias=1.0)

  # conv3, relu
  with tf.name_scope('conv3') as scope:
    kernel = tf.Variable(tf.truncated_normal([3, 3, 256, 384],
                                             dtype=tf.float32,
                                             stddev=1e-1), name='weights')
    conv = tf.nn.conv2d(lrn2, kernel, [1, 1, 1, 1], padding='SAME')  # Other conv after conv1 have padding = kernel size / 2
    biases = tf.Variable(tf.constant(0.0, shape=[384], dtype=tf.float32),
                         trainable=True, name='biases')
    bias = tf.nn.bias_add(conv, biases)
    conv3 = tf.nn.relu(bias, name=scope)
    parameters += [kernel, biases]
    print_activations(conv3)

  # conv4, relu
  with tf.name_scope('conv4') as scope:
    kernel = tf.Variable(tf.truncated_normal([3, 3, 384, 384],
                                             dtype=tf.float32,
                                             stddev=1e-1), name='weights')
    conv = tf.nn.conv2d(conv3, kernel, [1, 1, 1, 1], padding='SAME')
    biases = tf.Variable(tf.constant(0.0, shape=[384], dtype=tf.float32),
                         trainable=True, name='biases')
    bias = tf.nn.bias_add(conv, biases)
    conv4 = tf.nn.relu(bias, name=scope)
    parameters += [kernel, biases]
    print_activations(conv4)

  # conv5
  with tf.name_scope('conv5') as scope:
    kernel = tf.Variable(tf.truncated_normal([3, 3, 384, 256],
                                             dtype=tf.float32,
                                             stddev=1e-1), name='weights')
    conv = tf.nn.conv2d(conv4, kernel, [1, 1, 1, 1], padding='SAME')
    biases = tf.Variable(tf.constant(0.0, shape=[256], dtype=tf.float32),
                         trainable=True, name='biases')
    bias = tf.nn.bias_add(conv, biases)
    conv5 = tf.nn.relu(bias, name=scope)
    parameters += [kernel, biases]
    print_activations(conv5)

  # pool5
  pool5 = tf.nn.max_pool(conv5,
                         ksize=[1, 3, 3, 1],
                         strides=[1, 2, 2, 1],
                         padding='VALID',
                         name='pool5')
  print_activations(pool5)

  # Reshape  
  pool5_shape = pool5.get_shape().as_list()
  reshape = tf.reshape(
      pool5,
      # Keep batch same but concat the rest
      [pool5_shape[0], pool5_shape[1] * pool5_shape[2] * pool5_shape[3]])

  # FC6, relu
  with tf.name_scope('fc6'):
    weights = tf.Variable(
        tf.truncated_normal([pool5_shape[1] * pool5_shape[2] * pool5_shape[3], 4096],
                            stddev=0.01),
        name='weights')
    biases = tf.Variable(tf.zeros([4096]),
                         trainable=True, name='biases')
    fc6_nodrop = tf.nn.relu(tf.matmul(reshape, weights) + biases)
    fc6 = tf.nn.dropout(fc6_nodrop, 0.5, noise_shape=None, seed=None, name=None)
    parameters += [weights, biases]
    print_activations(fc6)
    
  # FC7, relu
  with tf.name_scope('fc7'):
    weights = tf.Variable(
        tf.truncated_normal([4096, 4096],
                            stddev=0.01),
        name='weights')
    biases = tf.Variable(tf.zeros([4096]),
                         trainable=True, name='biases')
    fc7_nodrop = tf.nn.relu(tf.matmul(fc6, weights) + biases)
    fc7 = tf.nn.dropout(fc7_nodrop, 0.5, noise_shape=None, seed=None, name=None)
    parameters += [weights, biases]
    print_activations(fc7)
    
  # FC8
  with tf.name_scope('fc8'):
    weights = tf.Variable(
        tf.truncated_normal([4096, 1000],
                            stddev=0.01),
        name='weights')
    biases = tf.Variable(tf.zeros([1000]),
                         trainable=True, name='biases')
    fc8 = tf.matmul(fc7, weights) + biases
    parameters += [weights, biases]
    print_activations(fc8)
    
  #softmax
  softmax = tf.nn.softmax(fc8)
  parameters += [weights, biases]
  print_activations(softmax)

  return softmax, parameters


def time_tensorflow_run(session, target, info_string):
  """Run the computation to obtain the target tensor and print timing stats.

  Args:
    session: the TensorFlow session to run the computation under.
    target: the target Tensor that is passed to the session's run() function.
    info_string: a string summarizing this run, to be printed with the stats.

  Returns:
    None
  """
  num_steps_burn_in = 10
  total_duration = 0.0
  total_duration_squared = 0.0
  for i in xrange(FLAGS.num_batches + num_steps_burn_in):
    start_time = time.time()
    _ = session.run(target)
    duration = time.time() - start_time
    if i > num_steps_burn_in:
      if not i % 10:
        print ('%s: step %d, duration = %.3f' %
               (datetime.now(), i - num_steps_burn_in, duration))
      total_duration += duration
      total_duration_squared += duration * duration
  mn = total_duration / FLAGS.num_batches
  vr = total_duration_squared / FLAGS.num_batches - mn * mn
  sd = math.sqrt(vr)
  print ('%s: %s across %d steps, %.3f +/- %.3f sec / batch' %
         (datetime.now(), info_string, FLAGS.num_batches, mn, sd))



def run_benchmark():
  """Run the benchmark on AlexNet."""
  with tf.Graph().as_default():
    # Generate some dummy images.
    image_size = 227
    # Note that our padding definition is slightly different the cuda-convnet.
    # In order to force the model to start with the same activations sizes,
    # we add 3 to the image_size and employ VALID padding above.
    
    # BENCHMARK NOTE: Caffe and CcT read these from disk, whereas tensorflow
    # needs to generate floats but not go to disk.
    
    images = tf.Variable(tf.random_normal([FLAGS.batch_size,
                                           image_size,
                                           image_size, 3],
                                          dtype=tf.float32,
                                          stddev=1e-1))

    # Build a Graph that computes the logits predictions from the
    # inference model.
    softmax, parameters = inference(images)

    # Build an initialization operation.
    init = tf.initialize_all_variables()

    # Start running operations on the Graph.
    config = tf.ConfigProto()
    config.gpu_options.allocator_type = 'BFC'
    sess = tf.Session(config=config)
    sess.run(init)

    # Run the forward benchmark.
    time_tensorflow_run(sess, softmax, "Forward")

    # Add a simple objective so we can calculate the backward pass.
    objective = tf.nn.l2_loss(softmax)
    # Compute the gradient with respect to all the parameters.
    grad = tf.gradients(objective, parameters)
    # Run the backward benchmark.
    time_tensorflow_run(sess, grad, "Backward")


def main(_):
  run_benchmark()


if __name__ == '__main__':
  tf.app.run()
