package edu.stanford.peer.rbtm.engine;

import java.util.*;

import edu.stanford.peer.rbtm.RBTMConstants;
import edu.stanford.peer.rbtm.credential.*;
import edu.stanford.peer.rbtm.util.*;

/**
   An object of the class CredentialManager stores credentials and answer
   queries.
 */

public class CredentialManager 
  implements java.io.Serializable, ProofGraph, RBTMConstants
{
  static final public Iterator emptyIt = new ArrayList(1).iterator();
  static final public Collection emptyCollection = new ArrayList(1);

  // stores all the credentials
  private HashSet credentials = new HashSet();

  // an index from issuers to credential collections
  private HashMap indexByIssuer = new HashMap();

  // an index from roles to credential collections
  private HashMap indexByRole = new HashMap();

  // an index from subjects to collections of credentials
  private HashMap indexBySubject = new HashMap();

  private int track;

  private SolutionFilter solutionFilter;

  // a proof queue
  transient private ProofQueue proofQueue = new SimpleProofQueue();

  transient private HashMap    nodeTable = new HashMap();

  public CredentialManager(Set credSet, int trackType, 
			   SolutionFilter solFilter) 
  {
    this(trackType, solFilter);
    addCredentials(credentials);
  }

  public CredentialManager(int trackType, SolutionFilter solFilter) {
    track = trackType;
    solutionFilter = solFilter;
  }

  public CredentialManager() {
    this(TRACK_NONE, new DefaultSolutionFilter());
  }

  public void reinit() {
    proofQueue = new SimpleProofQueue();
    nodeTable = new HashMap();
  }

  synchronized public void addCredentials(Collection credentials) {
    reinit();
    Iterator credIt = credentials.iterator();
    while (credIt.hasNext()) {
      addCredential((StaticCredential)credIt.next());
    }
  }

  synchronized public void removeCredentials(Collection credentials) {
    reinit();
    Iterator credIt = credentials.iterator();
    while (credIt.hasNext()) {
      removeCredential((StaticCredential)credIt.next());
    }
  }

  synchronized public boolean addCredential(StaticCredential cred) {
    // Make sure that no duplicate Credential exists
    if (credentials.contains(cred)) {
      Debug.debugInfo("trying to add existing credential: " + cred);
      return false;
    }
    Debug.debugInfo("adding credential: " + cred);
    credentials.add(cred);
    addIndexFor(cred);
    ProofNode node = getNode(cred.getDefinedRole());
    if (node != null) {
      node.invalidateBackward();
    }
    node = getNode(cred.getSubject());
    if (node != null) {
      node.invalidateForward();
    }
    return true;
  }

  synchronized public void removeCredential(StaticCredential cred) {
    if (credentials.contains(cred)) {
      Debug.debugInfo("removing credential: " + cred);
      reinit();
      credentials.remove(cred);
      removeIndexFor(cred);
    } else {
      Debug.debugInfo("trying to remove nonexistent credential: " + cred);
      //// throw Exception
    }
  }

  private void addIndexFor(StaticCredential cred) 
  {
    Entity e = cred.getIssuer();
    ArrayList credList = (ArrayList) indexByIssuer.get(e);
    if (credList == null) {
      credList = new ArrayList(40);
      indexByIssuer.put(e, credList);
    }
    credList.add(cred);

    // Code to deal with indexByRole
    Role r = cred.getDefinedRole();
    credList = (ArrayList) indexByRole.get(r);
    if (credList == null) {
      credList = new ArrayList(10);
      indexByRole.put(r, credList);
    }
    credList.add(cred);

    EntityExpression subject = cred.getSubject();
    credList = (ArrayList) indexBySubject.get(subject);
    if (credList == null) {
      credList = new ArrayList(10);
      indexBySubject.put(subject, credList);
    }
    credList.add(cred);
  }

  private void removeIndexFor(StaticCredential cred) 
  {
    ((ArrayList)indexByIssuer.get(cred.getIssuer())).remove(cred);
    ((ArrayList)indexByRole.get(cred.getDefinedRole())).remove(cred);
    ((ArrayList)indexBySubject.get(cred.getSubject())).remove(cred);
  }

  /*
    public void addRole(Role role) {
    if (! indexByRole.containsKey(role)) {
    indexByRole.put(role, new ArrayList());
    }
    }
  */

  synchronized public Collection getCredentialsIssuedBy(Entity e) {
    Collection credentials = (Collection)indexByIssuer.get(e);
    if (credentials == null) {
      return emptyCollection;
    } else {
      return credentials;
    }
  }

  synchronized public Iterator findCredentialsDefiningRole(Role r) 
  {
    ArrayList credList = (ArrayList) indexByRole.get(r);
    if (credList == null) {
      return emptyIt;
    } else {
      return credList.iterator();
    }
  }

  synchronized public Iterator findCredentialsBySubject(EntityExpression subject)
  {
    ArrayList credList = (ArrayList) indexBySubject.get(subject);
    if (credList == null) {
      return emptyIt;
    } else {
      return credList.iterator();
    }
  }

  /*    
	boolean hasCredentialsForSubject(EntityExpression re) {
        return findCredentialsBySubject(re).hasNext();
	}
  */

  /**
   * Find out all Roles that roleExp belongs to.
   *
   * @param roleExp the entity expression that, whenever it is on the right
   * hand side of a credential statement, the left hand side will be returned.
   */
  synchronized public ResultEvidenceMap 
    forwardSearch(EntityExpression roleExp) {
    Debug.debugInfo("Brian: CredentialManager.forwardSearch");
    ProofNode goalNode = addForwardNode(roleExp);
    while (proofQueue.hasUnexploredForwardNodes()) {
      proofQueue.nextUnexploredForwardNode().forwardProcess();
    }
    return goalNode.getForwardSolutions();
  }

  // Find out all members of roleExp
  synchronized public ResultEvidenceMap backwardSearch(EntityExpression roleExp) {
    ProofNode goalNode = addBackwardNode(roleExp);
    while (proofQueue.hasUnexploredBackwardNodes()) {
      proofQueue.nextUnexploredBackwardNode().backwardProcess();
    }
    return goalNode.getBackwardSolutions();
  }

  public ProofNode addForwardNode(EntityExpression roleExp) {
    ProofNode node = addNode(roleExp);
    proofQueue.addForwardNode(node);
    return node;
  }

  public ProofNode addBackwardNode (EntityExpression roleExp) {
    ProofNode node = addNode(roleExp);
    proofQueue.addBackwardNode(node);
    return node;
  }

  /**
   * Add a node corresponding to a role expression, make sure that the 
   * node is not added twice. 
   * @param roleExp the expression associated with the node to be added.
   * @return the node associated with that expression. This may not be a new
   * node, if it was already present in the node table.
   */
  private ProofNode addNode (EntityExpression roleExp) {
    ProofNode node = (ProofNode)nodeTable.get(roleExp);
    if (node == null) {
      node = createProofNode(roleExp);
      nodeTable.put(roleExp, node);
    }
    return node;
  }

  private ProofNode getNode(EntityExpression roleExp) {
    return (ProofNode)nodeTable.get(roleExp);
  }
  
  /**
   * Create a proof node of the appropriate type. That is, it tries to figure
   * out what type of element roleExp really is, and create an appropriate
   * ProofNode for it.
   * 
   * @param roleExp an expression that will be the basis for the new node. 
   */
  private ProofNode createProofNode (EntityExpression roleExp) {
    if (roleExp instanceof Role) {
      return new RoleProofNode(this, (Role)roleExp, track);
    } else if (roleExp instanceof LinkedRole) {
      return new LinkedRoleProofNode(this, (LinkedRole)roleExp, track);
    } else if (roleExp instanceof Intersection) {
      return null; //new IntersectionProofNode(this, (Intersection)roleExp, track);
    } else {
      return new EntityProofNode(this, (Entity)roleExp, track);
    }
  }

  public ForwardSolution getForwardSolution(EntityExpression re) {
    return solutionFilter.getForwardSolution(re);
  }

  public BackwardSolution getBackwardSolution(EntityExpression re) {
    return solutionFilter.getBackwardSolution(re);
  }

  public String toString() {
    return "credentials=" + credentials.toString() + "\n" +
      "indexByIssuer=" + indexByIssuer.toString() + "\n" +
      "indexByRole=" + indexByRole.toString() + "\n" +
      "indexBySubject=" + indexBySubject.toString();
  }
	
  private String getSubjectIndexs()
  {
    StringBuffer outBuf = new StringBuffer();
    Iterator keyIt = indexBySubject.keySet().iterator();
    while (keyIt.hasNext()) {
      Object o = keyIt.next();
      if (o instanceof Role) {
	outBuf.append("Role ").append(o.toString()).append("\n");
	Role r = (Role) o;
	outBuf.append("Base is ").append(r.getBase().getClass()).append("\n");
	outBuf.append("RoleName is ").append(r.getName().getClass()).append("\n");
      }
    }
    return outBuf.toString();
  }

  /*
    protected void addProofEdge(ProofNode from, ProofNode to, Object e) {
    from.addChild(to, e);
    to.addParent(from, e);
    }
  */
  /*
    protected ProofEdge addProofEdge(ProofNode from, ProofNode to, Object e) {
    ProofEdge edge = new ProofEdge(from, to);
    if (! edgeTable.containsKey(edge)) {
    edgeTable.put(edge, new SmallSet(evidence));
    } else if (track) {
    ((SmallSet)edgeTable.get(edge)).add(e);
    }
    return edge;
    }
  */

  /*
    class GoalMonitor implements SolutionListener
    {
    private EntityExpression goalRE;
    GoalMonitor(EntityExpression goal) { goalRE = goal; }
    public void solutionAdded(ProofGraph.ProofNode s,  EntityExpression re) {
    if (goalRE.equals(re)) {
    throw new GoalReachedException(s, re);
    }
    }
    }
  */

}
