/*
 * StaticCredential.java
 */

package edu.stanford.peer.rbtm.credential;

import java.util.*;

/**
 * StaticCredential is a credential of the form role <- subject. The role is
 * of type 'Role' (formed by an entity, and a rolename), and subject is of
 * type 'EntityExpression'. 
 *
 * @see Role
 * @see EntityExpression
 */
public class StaticCredential extends Credential
{
  private Role      role;
  private EntityExpression subject;

  public StaticCredential(Role r, EntityExpression s) { 
    role = r;  
    subject = s; 
  }

  public StaticCredential(Entity issuer, RoleName n, EntityExpression s) { 
    role = new Role(issuer, n);
    subject = s; 
  }

  public StaticCredential(String e, String rn, String es) 
    throws CredentialParsingException  
  {
    role = new Role(new SimpleEntity(e), new SimpleRoleName(rn));
    subject = StaticCredential.getEntityExpression(es);
  }

  public Entity getIssuer()       { return role.getBase(); }

  //public Role   getRole()         { return role; }

  public Role   getDefinedRole()         { return role; }

  public EntityExpression getSubject()   { return subject; }

  public String toString() {
    return role.toString() + "<-" + subject.toString();
  }

  public int hashCode() {
    return role.hashCode() * 2 + subject.hashCode();
  }

  public boolean equals(Object o) 
  {
    return (o instanceof StaticCredential) &&
      role.equals(((StaticCredential)o).role) &&
      subject.equals(((StaticCredential)o).subject);
  }

  // convert a String into an EntityExpression, only deals with Role and LinkedRole for now
  static EntityExpression getEntityExpression(String str) throws CredentialParsingException {
    int i = 0;
    String[] names = new String[3];
    int n = 0;
    while (i<str.length()) {
      int j=str.indexOf(".", i);
      if (j == -1) {
	j = str.length();
      }
      names[n++] = str.substring(i, j).trim();
      i = j + 1;
    }
    if (n == 0) {
      throw new CredentialParsingException("can't parse EntityExpression from '" + str + "'");
    } else if (n == 1) {
      return new SimpleEntity(names[0]);
    } else if (n == 2) {
      return new Role(names[0], names[1]);
    } else if (n == 3) {
      return new LinkedRole(names[0], names[1], names[2]);
    } else {
      throw new CredentialParsingException("can't parse EntityExpression from '" + str + "'");
    }
  }

}
