/*
    Beforeing running this script to create database tables,
    "database_name" should be replaced with the value for the context 
    parameter "DatabaseName" defined in the web.xml
*/



CREATE TABLE IF NOT EXISTS database_name.users
(
    uid             int(4) NOT NULL AUTO_INCREMENT,
    uname           varchar(50) NOT NULL  UNIQUE,
    activeEmail     varchar(50) NOT NULL  UNIQUE,
    organization    longtext,
    city            longtext,
    state           varchar(2),
    homeLockerID    int(4) NOT NULL default 0,
    homeLockerQuota int(11) NOT NULL default 0,
    usertype        enum('full', 'readonly', 'denied', 'blocked') default 'readonly',
    pass            varchar(20),
    rlogin          enum('yes','no') DEFAULT 'yes',
    change_pass     enum('yes','no') DEFAULT 'yes',
    PRIMARY KEY (uid)
);

CREATE TABLE IF NOT EXISTS database_name.emails
(
   uid              int(4) NOT NULL,
   email            varchar(50) NOT NULL UNIQUE,
   PRIMARY KEY (uid, email),
   FOREIGN KEY (uid) REFERENCES database_name.users
);

/*
    uid             --  the issuer's uid.
    left_rolename   --  the rolename.

    uid and left_rolename together determine the credential(Role)
    this permission refers to. 
    
    right_entityid  --  the uid of the user belonging to this role definition
*/
CREATE TABLE IF NOT EXISTS database_name.credentials
(
    credid          int(4) NOT NULL AUTO_INCREMENT,
    uid             int(4) NOT NULL,        
    left_rolename   varchar(50) NOT NULL, 
    right_entityid  int(4) NOT NULL,
    right_rolename1 varchar(50) NOT NULL,
    right_rolename2 varchar(50) NOT NULL,
    PRIMARY KEY (credid),
    FOREIGN KEY (uid) REFERENCES database_name.users,
    FOREIGN KEY (right_entityid) REFERENCES database_name.users
);

/*
    issuerid and rolename together determine the credential(Role)
    this permission refers to.  
    rolename references to left_rolename in credentials table.
    
    receiver is the group that is given this permission.
*/
CREATE TABLE IF NOT EXISTS database_name.credential_permissions
(
    permid          int(4) NOT NULL AUTO_INCREMENT,
    issuerid        int(4) NOT NULL,        
    rolename        varchar(50) NOT NULL, 
    receiver        varchar(50) NOT NULL,
    read_perm       enum('yes', 'no') NOT NULL DEFault 'no', 
    PRIMARY KEY (permid),
    FOREIGN KEY (issuerid) REFERENCES database_name.users
);

/*
    owner       --  the uid of the favorite owner.
*/   
CREATE TABLE IF NOT EXISTS database_name.favorites
(
    favorname       varchar(100) NOT NULL,
    owner           int(4) NOT NULL,       
    path            varchar(255) NOT NULL,
    create_date     timestamp(14) NOT NULL,
    PRIMARY KEY (owner, path),
    FOREIGN KEY (owner) REFERENCES database_name.users
);

/*
    owner           -- the uid of the locker's owner.
    parent_locker   -- the lockerid of the parent locker.
    max_quota       -- the maximum quota allowed for this locker
    guarenteed_quota    -- the quota which is actually allocated
    actual_size     -- the actual space used in this locker
*/
CREATE TABLE IF NOT EXISTS database_name.lockers
(
    lockerid        int(4)  NOT NULL AUTO_INCREMENT,
    lockername      varchar(100) NOT NULL,
    owner           int(4) NOT NULL,        
    parent_locker   int(4) NOT NULL,       
    max_quota       int(11) NOT NULL DEFAULT 0,
    guaranteed_quota int(11) NOT NULL DEFAULT 0,
    actual_size     int(11) NOT NULL DEFAULT 0,
    modify_date     timestamp(14),
    create_date     timestamp(14),
    PRIMARY KEY (lockerid),
    FOREIGN KEY (owner) REFERENCES database_name.users,
    FOREIGN KEY (parent_locker) REFERENCES database_name.lockers
    
);

/*
    entityid        -- the uid of the permission issuer.
    rolename        -- the permission receiver and it references to 
                       the "left_rolename" in credential table.
    upload_perm     -- the permission to upload newer version.
    quota           -- the given quota for creating sublockers
*/    
CREATE TABLE IF NOT EXISTS database_name.locker_permissions
(
    permid          int(4) NOT NULL AUTO_INCREMENT,
    lockerid        int(4)    NOT NULL,
    entityid        int(4)    NOT NULL,        
    rolename        varchar(50) NOT NULL,              
    read_perm       enum('yes', 'no') DEFAULT 'yes',
    delete_perm     enum('yes', 'no') DEFAULT 'yes',
    sublocker_perm  enum('yes', 'no') DEFAULT 'yes',
    quota           int(11) NOT NULL DEFAULT 0,
    upload_perm     enum('yes', 'no') DEFAULT 'yes',    
    PRIMARY KEY (permid),
    FOREIGN KEY (entityid) REFERENCES database_name.users,
    FOREIGN KEY (lockerid) REFERENCES database_name.lockers

);

/*
    This table stores the most recent version of a file.
    owner -- the owner of the original file
    author -- the author name of the different versions of the file
*/
CREATE TABLE IF NOT EXISTS database_name.files
(
    fileid          int(4) NOT NULL AUTO_INCREMENT,
    filename        varchar(100) NOT NULL,
    owner           int(4) NOT NULL,
    author          longtext NOT NULL,
    locker          int(4) NOT NULL,
    mimetype        varchar(50) DEFAULT 'text/plain',
    size            int(11) NOT NULL DEFAULT 0,
    modify_date     timestamp(14),
    upload_date     timestamp(14),
    raccess enum('yes', 'no') DEFAULT 'yes',
    PRIMARY KEY (fileid),
    FOREIGN KEY (owner) REFERENCES database_name.users,
    FOREIGN KEY (locker) REFERENCES database_name.lockers
    
);

/*
    entityid        -- the uid of the permission issuer.
    rolename        -- the permission receiver and it references to 
                       the "left_rolename" in credential table.
    upload_perm     -- the permission to upload newer version.
*/    
CREATE TABLE IF NOT EXISTS database_name.file_permissions
(
    permid          int(4) NOT NULL AUTO_INCREMENT,
    fileid          int(4) NOT NULL,
    entityid        int(4) NOT NULL,        
    rolename        varchar(50) NOT NULL,            
    read_perm       enum('yes', 'no') DEFAULT 'yes',
    delete_perm     enum('yes', 'no') DEFAULT 'yes',
    upload_perm     enum('yes', 'no') DEFAULT 'no', 
    raccess         enum('yes', 'no') DEFAULT 'yes',
    PRIMARY KEY (permid),
    FOREIGN KEY (entityid) REFERENCES database_name.users,
    FOREIGN KEY (fileid) REFERENCES database_name.files
    
);

CREATE TABLE IF NOT EXISTS database_name.versions
(
    versionid       int(8) NOT NULL AUTO_INCREMENT,
    fileid          int(4)  NOT NULL,
    filename            longtext  NOT NULL,
    author          longtext NOT NULL,
    mimetype        varchar(50) DEFAULT 'text/plain',
    date            longtext,
    size           int(11) NOT NULL DEFAULT 0, 
    PRIMARY KEY (versionid),
    FOREIGN KEY (fileid) REFERENCES database_name.files
);

/*
    This table is used by UStorit Admin to enforce the rules 
    for locker creation.  Users will be given different home 
    locker max quota based on what email domains they are from.
    And if the homeLocker_quota is set to be zero, then users 
    cannot create locker but they will have accounts. If the quota
    value is -1, then they are denied the right to access UStorit.
    
    Another thing is that changes in rules only affect future users and
    no attempt should be made to update the usertype and 
    home locker quota of an existing user.
*/
CREATE TABLE IF NOT EXISTS database_name.rules
(
    email_domain        varchar(50) NOT NULL UNIQUE,
    homelocker_quota    int(11) NOT NULL DEFAULT -1,
    PRIMARY KEY (email_domain, homelocker_quota)
);

-- insert a root locker
INSERT INTO database_name.lockers (lockername, owner, parent_locker, max_quota, guaranteed_quota, actual_size, modify_date, create_date) VALUES ('rootlocker', 1, 1, 1000000, 0, 0, NOW(), NOW()); 

-- insert a root user (ustorit admin).
INSERT INTO database_name.users (uname, activeEmail, organization) VALUES ('rootuser', 'root@ustorit', 'UstoritAdmin');
-- insert a dummy admin's email address. The real admin email should be configured in web.xml
INSERT INTO database_name.emails(uid, email) VALUES('1', 'admin@ustorit');

-- insert rules
INSERT INTO database_name.rules(email_domain, homelocker_quota) VALUES('stanford.edu', 10000);
INSERT INTO database_name.rules(email_domain, homelocker_quota) VALUES('hotmail.com', 0);
INSERT INTO database_name.rules(email_domain, homelocker_quota) VALUES('theory.stanford.edu', 2000);
INSERT INTO database_name.rules(email_domain, homelocker_quota) VALUES('edu', 5000);
INSERT INTO database_name.rules(email_domain, homelocker_quota) VALUES('yahoo.com', -1);

SELECT * FROM database_name.lockers;
SELECT * FROM database_name.users;
SELECT * FROM database_name.rules;
