package jinifilter;

import java.net.*;
import java.io.*;

/*
  creating a SafeURLClassLoader solves the problem of intercepting the network code
  only if the service proxy object uses URLClassLoader as its net-class-loading mechanism
  if this is not true, and if RMIClassLoader does not use URLClassLoader inside its own code
  then we should opt for replacing the ClassLoader.defineClass() method through the
  Java Method Filter

* sun.rmi.server.LoaderHandler implements all the static calls in
  RMIClassLoader and also in java.rmi.MarshalledObject (which subclasses
  its input stream from sun.rmi.server.MarshalInputStream which is a 
  subclass of java.io.ObjectInputStream, overriding resolveClass(),
  which uses LoaderHandler.loadClass(ulr, name) to load the class ).
  Now the interesting part is that the LoaderHandler uses a private class
  Loader ( sun.rmi.server.LoaderHandler$Loader ) which is a subclass of 
  URLClassLoader without overriding findClass(). So, all of the RMIClasLOader
  situation should be resolved (even if we don't go for redirecting all
  ClassLoader.defineClass() ) simply by changing URLClassLoader refs to 
  SafeURLClassLoader. 

*/

public class SafeURLClassLoader extends URLClassLoader 
{
 	private static boolean weAreDebugging = false;
	private static void debug( String text )
	{
			if (weAreDebugging)
					System.out.println( text );
	}

	protected static ClassFilter classFilter = new ClassFilter();
	protected static MethodFilter methodFilter = new MethodFilter();

  public SafeURLClassLoader(URL[] urls) 
  { 
    super( urls ); 
    debug("Someone created " + this );
  }

  public SafeURLClassLoader(URL[] urls, ClassLoader parent) 
  { 
    super( urls, parent ); 
    debug("Someone created " + this );
  }
  
  public SafeURLClassLoader(URL[] urls, ClassLoader parent, URLStreamHandlerFactory factory) 
  {
    super( urls, parent, factory );
    debug("Someone created " + this );
  }


	/* loadClass and findClass are declared 'final' so that noone can 
		 override their behavior in a subclass
	*/
  public final Class loadClass(String name) throws ClassNotFoundException
  { 
     debug( name + " entering in loadClass() in " + this );
     Class cl = super.loadClass( name );
     debug( name + " exiting loadClass() in " + this );
     debug( "" );
     return cl;
  }

  // overriding findClass in URLClassLoader

  protected final Class findClass(String name) throws ClassNotFoundException
  { 
    debug( name + " in findClass() in " + this );


    byte[] bytes = null;
 
    try {
        // the network-downloaded files will normally not be found 
				// by the parent loader called by loadClass 
				bytes = loadClassData( name );  // it can handle JARs too !
    } catch ( Exception e ) {
				//e.printStackTrace( System.out );
				throw new ClassNotFoundException();
    }


    byte[] modBytes = bytes;

    /* 
      All classes must be modified here since it means that the 
			delegation failed and we had to pull it from the network.
    */
      

    modBytes = classFilter.modifyClass( name, modBytes );
    modBytes = methodFilter.modifyClass( name, modBytes );

    debug("");

    return defineClass( name, modBytes, 0, modBytes.length );	 
  }



  // helper function
  private byte [] loadClassData( String name ) throws IOException
  {
    name = name.replace( '.', '/' );
    name += ".class";

		/* When I run the JiniFilter, this code is used to get the bytecode
			 for all remote files. Then, the call to this method results 
			 in two HTTP requests to the LookupServer to get the same .class file.
			 When I run the app by itself, we are not using getResource() to
			 get to the class but instead, whatever the original 
			 URLClassLoader.findClass implementation is -- and there is only
			 one call per .class file. I don't know why there is such a difference.
		*/
    InputStream ins = getResourceAsStream( name );
    if ( ins == null ) {
				debug( "Could not find " + name );
				throw new IOException();
    }

    //debug("We found " + name +".");

    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    byte buff[] = new byte[100];

    while( true ) {
      int r = 0;	
      r = ins.read( buff );
      if ( r == -1 )
	 break;
      bos.write( buff, 0, r );
    }

    byte resultBuff[] = bos.toByteArray();
    
    ins.close();
    bos.close();    

    return resultBuff;  
  } 
}
