package jinifilter;

/*
	The idea behind this jinifilter package is that the application
	programmer can write a normal app using the standard JINI framework,
	and then he/she can run it with our package which will take care
	of all the security-motivated modifications, transparently,
	without requiring changes to the original program. 

	There are two basic principles ruling the modifications of classes
	that this code implements:
	
	1) if class A (remote or local-jini) is loading other classes, 
	   it will be modified so that its loading mechanism is replaced 
		 by ours,
 
	2) if class A (remote) is using one of the classes listed in 
	   classnames.txt or methodnames.txt, it will be modified so that 
		 these references are replaced with 'Safe' refs.
*/

/*
  Notice that here in SafeClassLoader, we only replace references
  to the remote loading mechanism (in the current implementation, 
	java.net.URLClassLoader) and not those to java.lang.ClassLoader. 
	We don't have to worry about the referenced local classes
	since the specs of the JVM require that if a class A loaded by L, 
	references a class B, the JVM will try to locate a version of B as 
	loaded with L, and if there isn't one, it will ask L to try to load it.
	This is why passing all of the local classes referenced by the main jini 
	app through our SafeClassLoader is not a problem (since we loaded the 
	main app with this loader, and also, we have been careful which classes we 
	delegate for loading to our parent, the bootstrap loader). In other 
	words, we get the local safe-loading for free. Where we run into a 
	problem is when we have to deal with remote clases. In this case, 
	the JVM will pass to SafeClassLoader their fully qualified names 
	but we won't know the codebase url (known only to the app level class).
	Our solution to this problem is (in this implementation) to replace
	all the references in the local classes to java.net.URLClassLoader with
	'jinifilter.SafeClassLoader'. This is the rational for doing so: 
	if the jini app takes advantage of the RMI framework for downloading
	classes (which because of its great convenience, we assume 
	in this implementation that it will, but we don't have to), then at some
  point the request for the bytecode of the remote class will pass through
	an instance of a private subclass of java.net.URLClassLoader 
	(sun.rmi.server.LoaderHandler$Loader). If it doesn't, we assume that
	the jini user app would then use an instance of java.net.URLClassLoader. 
	The second case is obvious -- after modification, the remote class loading
	will be done throgh 'SafeURLClassLoader'. The first case is a bit more 
	interesting because the RMI routines use a subclass of URLClassLoader, and
	not an instance. This will obviously be a problem, if they have overriden
	the findClass or the loadClass methods (loadClass calls findClass if the
	classname could not be handled by the parent, and findClass does all of
	the remote-downloading work) since SafeURLClassLoader overrides findClass
	to insert the modification step between the bytecode downloading and 
	definition. Fortunately, they are relying on the parent's findClass 
	functionality, which means that their instance will still call our
	SafeURLClassLoader.findClass() code if we modify the 
	sun.rmi.server.LoaderHandler$Loader.
*/


import java.lang.*;
import java.io.*;
import java.net.*;
import java.security.*;
import java.util.*;
import java.lang.reflect.*;

public class SafeClassLoader extends ClassLoader 
{
	private static boolean weAreDebugging = false;
	private static void debug( String text )
	{
			if (weAreDebugging)
					System.out.println( text );
	}


		/* 
			 When we use the filters in SafeClassLoader, we should not use
			 the full modification power of the filters (replacing all
			 the references in the 'classnames' and 'methodnames' files) since
			 these are only needed for the remote classes (those that 
			 SafeClassLoader cannot load as it is -- without subclassing). 
			 This is why we use this special filter constructor (taking a String
			 argument) which causes the filters to ignore their usual config files.

			 Note that we need to replace just a single reference -- 
			 either URLClassLoader (which we do in this implementation),
			 or ClassLoader.defineClass() -- but not both at the same time.
			 We don't need a method filter, if we are relying on the
			 omnipresence of URLClassLoader as a loading mechanism
			 (which is the chosen approach of this implementation).

			 In case we decide to go with redirecting ClassLoader.defineClass,
			 we can use something like this:
			 
			 filter = new MethodFilter( "java/lang/ClassLoader.defineClass" );
		*/
	protected static ClassFilter filter = 
			new ClassFilter( "java/net/URLClassLoader" );



	public SafeClassLoader ( ) {
			super( );
			debug("Someone created " + this );
  }

  public SafeClassLoader ( ClassLoader parent ) {
			super( parent );
			debug("Someone created " + this );
  }

  protected boolean shouldDelegate( String name ) 
  {    
			//System.out.println( "Testing '" + name + "' for delegation ..." );
			
			// have to load it ourselves, if we intend to modify it
			if ( shouldModify( name ) )
					return false;

			/* 
				 In order to modify the classes we want to modify, we need to load
				 them with our loader. The problem is that as each class is loaded,
				 all the classes that it references are attempted to be resolved with
				 that class' class loader. The Java class loading model
				 relies on proper delegation -- first let the parent loader try to
				 load the class, then if it fails, you try to load it.
				 The problem here is that we can't load all classes with our loader
				 just so that we are on the safe side -- we are running in linking
				 problems with some java.* classes. 
				 So, the alternative is to figure out who is referencing who and 
				 load only the necessary ones.
				 For the user apps, the chain of class loading is pretty clear:
				 Tester -> LookupLocator -> ServiceRegistrar obj -> service obj
				 but this is not the case with the most important
				 sun.rmi.server.LoaderHandler$Loader which does all of the remote
				 class loading internally in this java implementation 
				 (Java 2 ver 1.2.2 for Win32).
				 So, we better load all java.rmi.* classes which we'll use since they 
				 may reference sun.rmi.server.LoaderHandler which will reference
				 sun.rmi.server.LoaderHandler$Loader (its a private class declaration).
			*/
			if ( name.startsWith( "java.rmi." ) ||
					 name.startsWith( "sun.rmi.server." ) 
					 )
					return false;
			
			return true;
  }

  protected boolean shouldModify( String name ) 
  {
			//System.out.println( "Testing '" + name + "' for modifications..." );
			/* 
				 none of the remote classes will be able to be loaded in this loader,
				 though, so we don't need to worry about them here --
				 they should be the worry of SafeURLClassLoader ...
				 Of course, this is the case only if we rely that URLClassLoader
				 will be the only mechanism used to get the remote files.
				 In the case when the bytecode is generated on the fly, or
				 a socket network connection is generated, the only point
				 of entry for the definition of the class becomes 
				 ClassLoader.defineClass() where now, we'll need to specify
				 explicitely what classes we need to modify (if for nothing else
				 then at least for practical reasons, since the ClassFilter and
				 MethodFilter seem to have problems with modifying certain classes...
				 
				 To recap: 
				      if we rely on URLClassLoader being the only entry point
				 for remote classes, then here, we should only take care of the
				 propagation of the loading hook (the 'Safe' version of 
				 URLClassLoader in the classes which we expect to attempt 
				 to load other classes (like jini.LookupLocator, which loads
				 the object implementing jini.ServiceRegistrar, which loads the 
				 service object). We have reasons to believe that this is going to 
				 be the case in this implementation of java, since it relies 
				 for all remote code downloads on sun.rmi.server.LoaderHandler$Loader
				 which is a sublass of URLClassLoader.
				      if we can't rely on that and we try to replace all calls to
				 ClassLoader.defineClass(), then it's not exactly clear whether or
				 not the calls will be dispatched to the right classes (e.g. a 
				 defineClass() call made in URLClassLoader which is a subclass of
				 ClassLoader -- does the MethodFilter change references in such a way
				 so that SafeURLClassLoader.defineClass()/static or
				 SafeClassLoader.defineClass()/static gets called ? 
				 In any case, we can dispatch it appropriately based on the first 
				 argument in the static defineClass... The bottom line is that 
				 this is not clear at this point since we did not 
				 explore this path yet....				 
			*/

			/* 
				 The bottom line here is that we should modify ALL of the classes
				 which are on the path of references starting from the main app class,
				 and leading to ServiceRegistrar. ServiceRegistrar itself needs no
				 modification since it's only an interface; the class implementing it
				 will be downloaded from the net and will be automatically modified
				 by SafeURLClassLoader.
				 Since computing this path of references is rather complicated and
				 time-expensive, the alternative is to load and modify every class
				 outside of the JDK and the jinifilter package. Since the Java
				 Filter classes sometimes create trouble in their modification
				 business (bug? feature?), we can explicitely exclude particular
				 classes from being modified, if we are sure they are not on
				 the above mentioned reference path leading to ServiceRegistrar.

				 Notice that we are treating differently SafeClassLoader and 
				 SafeURLClassLoader -- this is because in the first case (with
				 the local files), we rely on the JVM behavior that just by loading
				 one class with our loader (SafeClassLoader), all classes it references
				 will be attempted to be loaded through the same loader.

				 Only now I see how much easier it would have been if I have pursued
				 the replacement of ClassLoader.defineClass()... No worries on
				 loading and delegation and modifications... Or maybe not...
				 Whatever, too late now.
			*/

			if ( 
					name.startsWith( "java." ) ||
					name.startsWith( "javax.swing." ) ||
					name.startsWith( "org.omg.CORBA." ) ||
					name.startsWith( "sun." ) && 
					         ! name.equals( "sun.rmi.server.LoaderHandler$Loader") || 
					name.startsWith( "jinifilter." )
					)
				  return false;
			else
					return true;

			/*
				this was the old version ...
				 
			if ( 
					// local main app loading LookupLocator
					name.equals( "jinifilter.Tester" ) ||  
					
					// local jini class loading the remote ServiceRegistrar object
					name.equals( "jini.LookupLocator" ) ||  
			
					// implements the RMI class loading
					name.equals( "sun.rmi.server.LoaderHandler$Loader")
					) 
					return true;
			
			return false;
			*/
	}

/* 
	 this overriden method has the following approach to class loading --
	 delegate only the system classes for the parent (the sysloader) to load since otherwise
	 (if we define them) they seem to make problems at link time.
	 all other files (all the user files) are to be modified and loaded (defined) with our loader
*/

		/* loadClass and findClass are declared 'final' so that noone can 
			 override their behavior in a subclass
		*/
		protected final Class loadClass(String name, boolean resolve) 
				throws ClassNotFoundException
		{
				debug( name + " in loadClass() in " + this );
				byte[] bytes;
				Class resultClass;
				
				if ( shouldDelegate( name ) ) {
						// call the normal loadClass that uses delegation 
						debug( "'" + name + "' is not our target. Delegating the loading to our parent..."); debug("");
						return super.loadClass( name, resolve );
				}
				
				resultClass = findLoadedClass( name );
				if ( resultClass != null) {
						debug(name + " was already loaded.");
						return resultClass;
				}
				
				resultClass = findClass( name );
				
				if (resolve) 
						resolveClass( resultClass );
				
				return resultClass;
		}
		
		protected final Class findClass(String name) throws ClassNotFoundException
		{ 
				debug( name + " in findClass() in " + this );
				
				byte[] bytes = null;
				
				try {
						bytes = loadClassData( name );  // it can handle JARs too !
				} catch ( Exception e ) {
						//e.printStackTrace( System.out );
						throw new ClassNotFoundException();
				}
				
				return modifyAndDefineClass( name, bytes );
		}
		
		
		public byte[] loadClassData( String name ) throws IOException 
		{
				name = name.replace( '.', '/' );
				name += ".class";
				
				InputStream ins = getResourceAsStream( name );
				if ( ins == null ) {
						debug( "Could not find " + name + " in getResourceAsStream()" );
						throw new IOException();
				}
				
				//System.out.println("We found " + name +".");
				
				ByteArrayOutputStream bos = new ByteArrayOutputStream();
				byte buff[] = new byte[100];
				
				while( true ) {
						int r = 0;	
						r = ins.read( buff );
						if ( r == -1 )
								break;
						bos.write( buff, 0, r );
				}
				
				byte resultBuff[] = bos.toByteArray();
				
				ins.close();
				bos.close();    
				
				return resultBuff;  
		}    
		
    /* 
			 it would have been nice of we could do this:
			 // by design, we always try to define the class with the original loader (our parent)
			 // this is so that we stay out of the way of the class loading definitions (to keep them
			 // undisturbed) only making sure that we are modifying the classes before they get defined
			 but, we can't since we cannot call a protected method on an object other than _this_ ...
			 was the semantics of protected the same in C++ or i am forgetting stuff ? whatever.
			 
			 ClassLoader originalLoader = getParent();
			 if ( originalLoader == null) 
			 originalLoader = ClassLoader.getSystemClassLoader();
			 return originalLoader.defineClass( name, b, off, len );
    */
		
		
		
		
		/*
			we need a public method that does the same as defineClass() so 
			that it can be called from the static method which is supposed to 
			replace the call to defineClass()...
		*/
		
		public Class modifyAndDefineClass( String name, byte[] b ) 
		{
				//System.out.println(name + " in modifyAndDefineClass called in " + this +" ("+ len+ " bytes)" );
				
				/*
					I tried using only the method filter to do also the job of the classfilter but it did not work
				*/ 
				
				byte[] modBytes = b;
				
				if ( shouldModify( name ) ) {
						modBytes = filter.modifyClass( name, modBytes );
				}
				debug("");
				
				return defineClass( name, modBytes, 0, modBytes.length );	 
		}
		
		





  public static Class defineClass( ClassLoader cl, String name, byte[] b, int off, int len)
			throws ClassFormatError {
			debug("Someone called defineClass in our loader for " + name );
			/* we need to get to the protected final defineClass() 
				 which is otherwise inaccessible from this static method,
				 and the method needs to be static because this is how 
				 the JavaMethodFilter works...
				 Madness.
			*/
			SafeClassLoader myloader = new SafeClassLoader( cl );
			
			// to make the byte array to start from 0
			if ( off != 0 ) {
					ByteArrayOutputStream bos = new ByteArrayOutputStream();
					bos.write( b, off, len );
					b = bos.toByteArray();    
			}
			return myloader.modifyAndDefineClass( name, b );
  }
				

}


/* 
	 what i should attempt in the future :
	 
	 * try to make only one loading-specific change to all classes --
	 
	 ** replace URLClassLoader refs with SafeURLClassLoader, hoping
	 that this is the only used remote class loading mechanism
	 (done)
	 
	 ** replace ClassLoader.defineClass with 
	 SafeClassLoader.defineClass/static. this way we don't need 
	 to worry about any other means of getting to the class bytecode.
	 and there will be no confusions and issues with who
	 loads what.
	 (left for the future)
*/







/* 
  Problems:

  * When we try to modify the MarshalInputStream class, we are running into some problems with the
	following static code (which the class initializer must process). 

    private final static boolean useCodebaseOnly =
		  ((Boolean) java.security.AccessController.doPrivileged(
            new GetBooleanAction("java.rmi.server.useCodebaseOnly"))).booleanValue();

	static {
	 try {
	    String monitor = sun.rmi.server.Activation.
		ActivationMonitorImpl.class.getName() + "_Stub";
	    String system = sun.rmi.server.Activation.
		ActivationSystemImpl.class.getName() + "_Stub";
	    String registry = sun.rmi.registry.
		RegistryImpl.class.getName() + "_Stub";
	    
	    permittedSunClasses.put(monitor, Class.forName(monitor));
	    permittedSunClasses.put(system, Class.forName(system));
	    permittedSunClasses.put(registry, Class.forName(registry));
	    
	 } catch (ClassNotFoundException e) {
	    throw new NoClassDefFoundError("Missing system class: " + 
					   e.getMessage());
	}
    }
    
		At some point, there is a reference to a class in this code (made during modifications 
		apparently, as the class is being defined) which cannot be resolved... and we get the 
		following sad, long exception:

		java.lang.ExceptionInInitializerError: java.lang.NullPointerException
		at MethodFilter.modifyInvokeInstrs(MethodFilter.java, Compiled Code)
		at MethodFilter.modifyClass(MethodFilter.java:454)
		at SafeClassLoader.modifyAndDefineClass(SafeClassLoader.java:235)
		at SafeClassLoader.findClass(SafeClassLoader.java:163)
		at SafeClassLoader.loadClass(SafeClassLoader.java, Compiled Code)
		at java.lang.ClassLoader.loadClass(ClassLoader.java, Compiled Code)
		at java.lang.ClassLoader.defineClass0(Native Method)
		at java.lang.ClassLoader.defineClass(ClassLoader.java, Compiled Code)
		at java.lang.ClassLoader.defineClass(ClassLoader.java:392)
		at SafeClassLoader.modifyAndDefineClass(SafeClassLoader.java:239)
		at SafeClassLoader.findClass(SafeClassLoader.java:163)
		at SafeClassLoader.loadClass(SafeClassLoader.java, Compiled Code)
		at java.lang.ClassLoader.loadClass(ClassLoader.java, Compiled Code)
		at java.lang.Class.forName0(Native Method)
		at java.lang.Class.forName(Class.java, Compiled Code)
		at sun.rmi.server.MarshalInputStream.class$(MarshalInputStream.java:39)
		at sun.rmi.server.MarshalInputStream.<clinit>(MarshalInputStream.java:73)
		at java.rmi.MarshalledObject.get(MarshalledObject.java:139)
		at ServiceRegistrar.lookup(Tester.java:86)
		at Tester.run(Tester.java:18)
		at SafeClassLoader.main(SafeClassLoader.java:300)
		Exception in thread "main" 
		

   and the loader transcript says:

	 sun.rmi.server.MarshalInputStream in loadClass() in SafeClassLoader@3f7ccec1
	 sun.rmi.server.MarshalInputStream in findClass() in SafeClassLoader@3f7ccec1
	 sun.rmi.server.MarshalInputStream is getting modified in ClassFilter@9cd0cec1
	 sun.rmi.server.MarshalInputStream is getting modified in MethodFilter@3d6ccecf
	 
	 java.io.ObjectInputStream in loadClass() in SafeClassLoader@3f7ccec1
	 'java.io.ObjectInputStream' is not our target. Delegating the loading to our parent...
	 
	 sun.security.action.GetBooleanAction in loadClass() in SafeClassLoader@3f7ccec1
	 'sun.security.action.GetBooleanAction' is not our target. Delegating the loading to our parent...
	 
	 java.lang.Boolean in loadClass() in SafeClassLoader@3f7ccec1
	 'java.lang.Boolean' is not our target. Delegating the loading to our parent...
	 
	 sun.rmi.server.Activation$ActivationMonitorImpl in loadClass() in SafeClassLoader@3f7ccec1
	 sun.rmi.server.Activation$ActivationMonitorImpl in findClass() in SafeClassLoader@3f7ccec1
	 sun.rmi.server.Activation$ActivationMonitorImpl is getting modified in ClassFilter@e4dccece
	 sun.rmi.server.Activation$ActivationMonitorImpl is getting modified in MethodFilter@e5fccece
	 
	 java.rmi.server.UnicastRemoteObject in loadClass() in SafeClassLoader@3f7ccec1
	 java.rmi.server.UnicastRemoteObject in findClass() in SafeClassLoader@3f7ccec1
	 java.rmi.server.UnicastRemoteObject is getting modified in ClassFilter@4c10cec1
	 Error in reading class stream or parsing stream (java.lang.NullPointerException!!!
	 java.rmi.server.UnicastRemoteObject is getting modified in MethodFilter@4e9ccec1
	 
	 java.rmi.server.RemoteServer in loadClass() in SafeClassLoader@3f7ccec1
	 java.rmi.server.RemoteServer in findClass() in SafeClassLoader@3f7ccec1
	 java.rmi.server.RemoteServer is getting modified in ClassFilter@2c44cec0
	 Error in reading class stream or parsing stream (java.lang.NullPointerException!!!
	 java.rmi.server.RemoteServer is getting modified in MethodFilter@231ccec0
	 
	 java.rmi.server.RemoteObject in loadClass() in SafeClassLoader@3f7ccec1
	 java.rmi.server.RemoteObject in findClass() in SafeClassLoader@3f7ccec1
	 java.rmi.server.RemoteObject is getting modified in ClassFilter@262ccec0
	 Error in reading class stream or parsing stream (java.lang.NullPointerException!!!
	 java.rmi.server.RemoteObject is getting modified in MethodFilter@2564cec0
	 
	 java.rmi.activation.ActivationMonitor in loadClass() in SafeClassLoader@3f7ccec1
	 java.rmi.activation.ActivationMonitor in findClass() in SafeClassLoader@3f7ccec1
	 java.rmi.activation.ActivationMonitor is getting modified in ClassFilter@a464cec1
	 java.rmi.activation.ActivationMonitor is getting modified in MethodFilter@3cc4cecf
	 java.lang.ClassNotFoundException in loadClass() in SafeClassLoader@3f7ccec1
	 'java.lang.ClassNotFoundException' is not our target. Delegating the loading to our parent...
	 
	 java.lang.ExceptionInInitializerError in loadClass() in SafeClassLoader@3f7ccec1
	 'java.lang.ExceptionInInitializerError' is not our target. Delegating the loading to our parent...
	 
	 java.lang.Exception in loadClass() in SafeClassLoader@3f7ccec1
	 'java.lang.Exception' is not our target. Delegating the loading to our parent...
	 
	 
   Note the inability of the ClassFilter to deal with some of the classes referenced in 
	 MarshalInputStream. This should not matter since we are returning the original
   bytecode in this case...

	 Questions: 
	 The problem is that ClassFilter and MethodFilter are loaded by the bootstrap loader, 
   and so every class they reference is being loaded through the bootstrap loader also...
	 I have to look further into this...

	 Work-arounds:
	 putting 					 
      name.startsWith( "java.rmi.") && !name.equals("java.rmi.activation.ActivationMonitor") ||
   in shouldModify() so that we avoid ActivationMonitor, but then we run into the same problem
   with java.rmi.server.RMIClientSocketFactory...

*/
