/**$*************************************************************
 *file: ClassFilter.java based on modifyClassFilter.java        *
 *filter to modify java byte code in jini.
 *author : Vijay Ganesh. (vganesh@theory.stanford.edu)          *
 *                                                              *
 *also see:                                                     *
 *JavaClass Library Docs.standard java libraries. java vm       *
 *Paper on 'JavaByteCode Modifcation by Inshik Shin'            *
 *other contact: amitp@cs.stanford.edu, mitchell@cs.stanford.edu*
 *                                                              *
 *License :                                                     * 
 *Copyright (C) 1994 Stanford University. All rights reserved.  * 
 *Permission is given to use, copy, and modify this software for*
 *any non-commercial purpose as long as this copyright notice is* 
 *not removed and the author's names are included in the copy-  *
 *right. All other uses, including redistribution in whole or in* 
 *part, are forbidden without prior written permission of the   * 
 *authors.                                                      * 
 *                                                              *
 *Disclaimer:                                                   *
 *This software is provided by the author AS IS.  The author    * 
 *DISCLAIMS any and all warranties of merchantability and fitnes* 
 *for a particular purpose.  In NO event shall the author be    *
 *LIABLE for any damages whatsoever arising in any way out of   * 
 *use of this software.                                         *
 ****************************************************************/
package jinifilter;

import java.io.*;
import java.util.Vector;
import java.util.Enumeration;
import java.lang.String;

import de.fub.bytecode.*;
import de.fub.bytecode.classfile.*;
import de.fub.bytecode.generic.*;
import de.fub.bytecode.util.*;


/****************************************************************
 *ClassFilter : class                                     *
 *                                                              *
 *filter which modifies the classes in a *.class file.          *  
 ****************************************************************/
public class ClassFilter { //implements DE.fub.inf.JVM.Constants {
 private static String catString = new String("jinifilter/Safe");

 private ConstantPool cpool;
 private Vector classNames;

	private boolean weAreDebugging = false;
	private void debug( String text )
	{
			if (weAreDebugging)
					System.out.println( text );
	}

  public ClassFilter()
  {
    classNames = new Vector();
    fillClassNames();
  }

	// ignoring the config file in which are listed all of the replacements
	// taking only one replacement
  public ClassFilter(String classname)
  {
    classNames = new Vector();
		classNames.addElement(classname);
  }


/****************************************************************
 *fillClassNames : method.                                      *
 *reads a file in pwd called 'classnames.txt' into the classNames  *
 *this file contains name of safe classes which should be added *
 *caveats:                                                      *
 *Exceptions handled:                                           *
 *filenot found.                                                *
 ****************************************************************/

 private void fillClassNames()
 {
   try {
		 FileInputStream f = new FileInputStream("jinifilter/classnames.txt");
		 //InputStream f  = this.getClass().getClassLoader().getResourceAsStream("classnames.txt");
BufferedReader fstream = new BufferedReader(new InputStreamReader(f));
    while(fstream.ready()) {
          classNames.addElement(fstream.readLine());
          //System.out.println(classNames.elementAt(0));
    }
    f.close();
    fstream.close();
   }
   catch(Exception e) {
        System.out.println("File 'classnames.txt' not found ( "+ e + " )");
   }
 }

/****************************************************************
 *setConstantPool:                                              *
 ****************************************************************/
 protected void setConstantPool(ConstantPool c) 
 {
    cpool = c;
 }

/****************************************************************
 *getConstantPool:                                              *
 ****************************************************************/
 protected ConstantPool getConstantPool() 
 {
     return cpool;
 }

/****************************************************************
 *modifyUtf8: method                                            *
 *                                                              *
 *this function goes thru' classNames Vector and modifies the   *
 *Utf8 entries.                                                 *
 ****************************************************************/

 protected void modifyUtf8() 
 {
   int count, len = cpool.getLength();
   
   for(Enumeration e = classNames.elements();e.hasMoreElements();) {
      String ss = new String((String)e.nextElement());	
    for(count = 1; count < len; count++) {
      Constant str = cpool.getConstant(count);
      //System.out.println(count + ": " + str.toString());
	   
      if(str.getTag() == 1) {
       if((((ConstantUtf8)str).getBytes()).equals(ss)) {
         ss = catString + ss.substring(ss.lastIndexOf("/")+1);
				 //System.out.println( ss );
         //((ConstantUtf8)str).setBytes(ss);
				 ConstantUtf8 y = new ConstantUtf8(ss);
         cpool.setConstant(count,y);
       }
      }
    }
   }

   /*DEBUG*/
/*
   for(count = 1; count < len; count++) {
       Constant str = cpool.getConstant(count);
       System.out.println(str.toString());
   }
*/
 }

 // don't modify the safe classes and the classes they substitute
 public boolean needsFiltering(String name)
 {
   for(Enumeration e = classNames.elements();e.hasMoreElements();) {
			 String classname = ((String)e.nextElement()).replace('/', '.').trim();	
			 String safename = catString.replace('/','.') + 
					 classname.substring(classname.lastIndexOf(".")+1).trim();
			 if ( name.equals( classname ) || name.equals( safename ) )
					 return false;
   }

   // for now, until MethodFilter and ClassFilter are in different classes,
   if (name.startsWith( catString ) ) {
			debug( name + " not in the modification list but looks 'Safe'");
			 return false;
   }
	 
   return true;
 }

   

  public byte[] modifyClass( String className, byte [] bytecode ) 
  {
     // don't modify the safe classes and the classes they substitute
     if ( ! needsFiltering( className ) )  { 
	System.out.println( className + " doesn't need modifications in " + this );
	return bytecode;
     }
     debug( className + " is getting modified in "+ this );
	
     ByteArrayInputStream codeInStream = 
		new ByteArrayInputStream ( bytecode );

     JavaClass presentClass = null;
     ClassParser parsedFile = null;
     File fileOfClassNames = null;

     try {
       parsedFile = new ClassParser(codeInStream, className);
       presentClass = parsedFile.parse();

       setConstantPool(presentClass.getConstantPool());
       modifyUtf8();
       presentClass.setConstantPool( getConstantPool() );

     } 
     catch(Exception e) {
	//e.printStackTrace( System.out );
	System.out.println("\t!!!Error in reading class stream or parsing stream ( " + e + " )");

	return bytecode; // return it as it was, untouched
     }

     // debug 	
     //presentClass.dump("/theory4/u1/vganesh/muffin.class");

     ByteArrayOutputStream modifiedCodeOutStream = new ByteArrayOutputStream ();
     try {
          presentClass.dump( modifiedCodeOutStream );
     } catch(Exception e) {
  	  System.out.println("error in creating outputstream");
     }

     return modifiedCodeOutStream.toByteArray();
  }
		
}
