// Sonny's solution to Hristo's earthquake
// October 1, 2008

#include <iostream>
#include <vector>
#include <algorithm>

using namespace std;

// some geometric structures
struct vec2 {
    int x, y;
    vec2(int xx = 0, int yy = 0) : x(xx), y(yy) { }
};

struct mat2 {
    int a, b, c, d;
    mat2(int aa = 1, int bb = 0, int cc = 0, int dd = 1) : a(aa), b(bb), c(cc), d(dd) { }
};

struct polygon {
    vector<vec2> verts;
};

// useful rotation matrices (about the origin)
const mat2 rotation[4] = 
{ mat2(), mat2(0, -1, 1, 0), mat2(-1, 0, 0, -1), mat2(0, 1, -1, 0) };

// operators on structures
bool operator<(const vec2 &a, const vec2 &b)
{
    if (a.y == b.y) return a.x < b.x;
    return a.y < b.y;
}

bool operator!=(const vec2 &a, const vec2 &b)
{
    return a.x != b.x || a.y != b.y;
}

vec2 operator+(const vec2 &a, const vec2 &b)
{
    return vec2(a.x + b.x, a.y + b.y);
}

vec2 operator-(const vec2 &a, const vec2 &b)
{
    return vec2(a.x - b.x, a.y - b.y);
}

vec2 operator*(const mat2 &m, const vec2 &v)
{
    return vec2(m.a*v.x + m.b*v.y, m.c*v.x + m.d*v.y);
}

// transforms the vertices of a polygon by matrix m
polygon operator*(const mat2 &m, const polygon &p)
{
    polygon q;
    for (int i = 0; i < p.verts.size(); ++i)
        q.verts.push_back(m * p.verts[i]);
    return q;
}

istream &operator>>(istream &stream, polygon &p)
{
    vec2 v, v1;
    stream >> v1.x >> v1.y;
    p.verts.push_back(v1);
    for (;;) {
        cin >> v.x >> v.y;
        if (!(v != v1)) break;
        p.verts.push_back(v);
    }

    return stream;
}

// translation and rotation invariant comparison of polygons
bool operator==(const polygon &p, const polygon &q)
{
    if (p.verts.size() != q.verts.size()) return false;
    int n = p.verts.size();

    // try all rotations
    for (int i = 0; i < 4; ++i) {
        polygon r = rotation[i]*q;
        const vector<vec2> &a = p.verts, &b = r.verts;

        // adjust for translation using top-left vertex
        int ai = min_element(a.begin(), a.end()) - a.begin();
        int bi = min_element(b.begin(), b.end()) - b.begin();
        vec2 t = a[ai] - b[bi];

        // compare
        bool good = true;
        for (int j = 0; j < n; ++j) {
            vec2 va = a[(ai+j) % n];
            vec2 vb = b[(bi+j) % n] + t;
            if (va != vb) { good = false; break; }
        }

        if (good) return true;
    }
    return false;
}

int main()
{
    int n;
    while (cin >> n)
    {
        if (n == 0) break;

        // read the broken glass and the window schematic
        vector<polygon> pieces(n), blueprint(n);
        for (int i = 0; i < n; ++i) cin >> pieces[i];
        for (int i = 0; i < n; ++i) cin >> blueprint[i];

        // compare each piece to schematic, assuming it's unique and exists
        for (int i = 0; i < n; ++i)
            for (int j = 0; j < n; ++j) {
                if (blueprint[j] == pieces[i]) {
                    if (i > 0) cout << ' ';
                    cout << j+1;
                    break;
                }
            }
        cout << endl;
    }

    return 0;
}