// Sonny's solution to Cargo Carriage
// (adapted from the original Fastest Fleet solution)
// October 1, 2008

#include <iostream>
#include <vector>
#include <queue>
#include <string>
#include <cctype>
#include <algorithm>

using namespace std;

enum direction { EW, NS };

// represents the functionality of a grid cell on the street map
struct node {
    char type;
    int ew, ns;
    
    node(char c = '.', int a = 0, int b = 0) : type(c), ew(a), ns(b) { }
    bool isValid() { return type != '.'; }
    
    // when the next green light will be
    int entryTime(int t, direction d)
    {
        if (type == '-') {
            int r = t%(ew+ns);
            if (d == EW && r >= ew) t += ew+ns-r;
            if (d == NS && r < ew)  t += ew-r;
        }
        else if (type == '|') {
            int r = t%(ew+ns);
            if (d == EW && r < ns)  t += ns-r;
            if (d == NS && r >= ns) t += ns+ew-r;
        }
        return t+1;
    }
};

// a search node
struct snode {
    int r, c, t;
    snode(int rr, int cc, int tt) : r(rr), c(cc), t(tt) { }
};

// for priority queue, lower t values should be "greater" so they come out first
bool operator<(const snode &a, const snode &b)
{
    return a.t > b.t;
}

// the workhorse class that handles keeping track of the street map
// and constructing a flow network through time from it
struct streets {
    
    int r, c;
    vector<string> raw;
    vector< vector<node> > nodes;
    int ar, ac;
    
    streets(int m, int n) : r(m), c(n) {
        raw = vector<string>(r);
        nodes = vector< vector<node> >(r, vector<node>(c));
    }
    
    // count the number of street lights
    int lights() {
        int n = 0;
        for (int i = 0; i < r; ++i)
            for (int j = 0; j < c; ++j)
                if (isdigit(raw[i][j]))
                    n = max(n, raw[i][j] - '0' + 1);
        return n;
    }
    
    // creates a node for a specified traffic light
    void setlight(char light, char dir, int ew, int ns) {
        for (int i = 0; i < r; ++i)
            for (int j = 0; j < c; ++j)
                if (raw[i][j] == light)
                    nodes[i][j] = node(dir, ew, ns);
    }
    
    // construct the nodes of our map
    void construct() {
        // create nodes for all cells on the raw map
        for (int i = 0; i < r; ++i)
            for (int j = 0; j < c; ++j) {
                if (!isdigit(raw[i][j]))
                    nodes[i][j] = node(raw[i][j]);
                // identify warehouse A where we start
                if (raw[i][j] == 'A') { ar = i; ac = j; };
            }
    }
                
    // run a shortest path algorithm on the streets
    int drive() {
        int sentry = 0x10000000;
        vector< vector<int> > seen(r, vector<int>(c, sentry));
        priority_queue<snode> pq;
        pq.push(snode(ar, ac, 0));
        
        // do the shortest path search!
        while (!pq.empty())
        {
            snode s = pq.top(); pq.pop();
            int i = s.r, j = s.c, t = s.t;
            
            // check for finishing condition
            if (nodes[i][j].type == 'B') return s.t;
            
            // if we got to i,j sooner than ever before, expand it
            if (s.t < seen[i][j]) {
                seen[i][j] = t;
                
                // then east-west
                if (j > 0 && nodes[i][j-1].isValid())
                    pq.push(snode(i, j-1, nodes[i][j-1].entryTime(t, EW)));
                if (j < c-1 && nodes[i][j+1].isValid())
                    pq.push(snode(i, j+1, nodes[i][j+1].entryTime(t, EW)));
                
                // and finally north-south
                if (i > 0 && nodes[i-1][j].isValid())
                    pq.push(snode(i-1, j, nodes[i-1][j].entryTime(t, NS)));
                if (i < r-1 && nodes[i+1][j].isValid())
                    pq.push(snode(i+1, j, nodes[i+1][j].entryTime(t, NS)));
            }
        }
        
        return -1;
    }
    
};

istream &operator>>(istream &stream, streets &s)
{    
    // read the map
    for (int i = 0; i < s.r; ++i)
        stream >> s.raw[i];
    
    // read the lights
    int n = s.lights();
    for (int i = 0; i < n; ++i) {
        int ew, ns;
        char light, dir;
        stream >> light >> dir >> ew >> ns;
        s.setlight(light, dir, ew, ns);
    }
    
    // construct the network
    s.construct();
    
    return stream;
}

int main()
{
    int m, n;
    while (cin >> m >> n)
    {
        if (m == 0 || n == 0) break;
        
        streets st(m, n);
        cin >> st;
        
        int time = st.drive();
        
        if (time > 0)   cout << time << endl;
        else            cout << "impossible" << endl;
    }
    
    return 0;
}
