#define __USE_BSD	
#include <sys/socket.h>	
#include <netinet/in.h>	
#include <netinet/ip.h>
#include <arpa/inet.h>
#define __FAVOR_BSD	
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include<netinet/ip_icmp.h>
#include <unistd.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
/** 
 * Header file containing the definitions and the required structures - 
 * Law - the rule specification to allow a particular transaction b/w src and dst
 * LawNode - the rule node in a tree. Each has a Law associated with it.
 * TreeNode - a node present in the LawTree. It has a linked list of LawNodes containing rules.
 * LawTree - the tree containing the laws and nodes.
 */
#include "firewall.h"

/** The tree having all the rules and nodes. */
static struct LawTree *lawtree; 

/** The words to be checked while content based filtering operation. numWords is their count.*/
static char **filter;
static int numWords = 0;

/** Law id specification */
static unsigned int counter = 0;

/** firewall active bit*/
int fw_active = 0; 

/** 
 * The combined structures for IP header and TCP/UDP Header. 
 * The individual structures are included in the header file
 *
 * <netinet/ip.h>
 * <netinet/tcp.h>
 * <netinet/udp.h>
 *
 * The exact attributes present in these structures can be refrenced 
 * form these header files.
 */
struct protoTCP {
	struct ip iphdr;
	struct tcphdr tcph;
};

struct protoUDP {
	struct ip iphdr;
	struct udphdr udph;
};

/**
 * The following function inserts a new law in the tree by searching the appropriate node. 
 * The tree is traversed using the SRC IP address present in the Law. If the SRC mask is 
 * used while specifying the law then an intermediate node is the tree shall contain
 * the law. The currentNode, so found, has a list of laws that contain the SRC address
 * ending at it. The new law is now added to this list.
 *
 * @param 	law - the law to be inserted in the tree
 * @return 	1 - in case of successful insertion in the tree
 * 	   	0 - in case of failure.
 */
static int insertLaw (struct Law *law) {
	if (law == NULL) {
		return 0;
	}

	IP_addr bit_post = 0x80000000;
	IP_addr mask = 0xFFFFFFFF;
	IP_addr address = law->src_addr;

	lawtree->root->prefix = (char *) malloc (1);
	lawtree->root->prefix = ".";	

	struct TreeNode *currentNode = lawtree->root;
	struct TreeNode *newNode;

	while ((bit_post != 0)) {
		if ((address & mask) == 0 && law->src_mask == 1) {
		      break;
		}
		  
		if ( (((address & bit_post) == 0) && (currentNode->left == NULL)) 
			|| (((address & bit_post) == bit_post) && (currentNode->right == NULL)) ) {
			
			newNode = (struct TreeNode *)malloc (sizeof (struct TreeNode));
			if (newNode == NULL) {
				return 0;
			}
			newNode->left = newNode->right = NULL;
			newNode->head = newNode->tail = NULL;
			newNode->parent = currentNode;
			newNode->prefix = (char *)malloc (32);
		    
			if ( (address & bit_post) == 0) {
				strcpy (newNode->prefix, currentNode->prefix);
				strcat (newNode->prefix, "0");
				currentNode->left = newNode;
			}
			else {
				strcpy (newNode->prefix, currentNode->prefix);
				strcat (newNode->prefix, "1");
				currentNode->right = newNode;
			}
		}

		/** 
		 * Traversing the left or right child depending on the next bit in the source IP address
		 */
		if ( (address & bit_post) == 0)
			currentNode = currentNode->left;
		else
			currentNode = currentNode->right;

		bit_post = bit_post >> 1;
		mask = mask >> 1;

	} // while loop ends
	//printf ("prefix = %s lawID = %d\n", currentNode->prefix, counter);
	
	/**
	 * The new LawNode to be inserted in the tree.
	 */
	struct LawNode *insertNode = (struct LawNode *)malloc (sizeof (struct LawNode));
	 
	if (insertNode == NULL) {
		return 0;
	}
	law->id = counter++;
	insertNode->law = law;
	insertNode->next = NULL;
	insertNode->prev = currentNode->tail;

	if (currentNode->head == NULL) {
		currentNode->head = insertNode;
		currentNode->tail = insertNode;
	} 
	else {
		currentNode->tail->next = insertNode;
		currentNode->tail = insertNode;
	}
	return 1;
}

/**
 * The following function deletes an existing lawnode present in the tree alongwith 
 * its list of law rules.
 * 
 * @param 	lawnode - the LawNode, present in the tree, that is to be deleted.
 * @return 	1 - in case of successful deletion from the tree
 *         	0 - in case of unsuccess.
 */
static int deleteLaw (struct LawNode *lawnode) {
	IP_addr bit_post = 0x80000000;
	IP_addr address = lawnode->law->src_addr;
	IP_addr mask = 0x11111111;
	struct TreeNode *currentNode = lawtree->root;
	struct TreeNode *tmp;
	while ((bit_post != 0)) {
		if (address & mask == 0) break;
			mask = mask >> 1;
			  
		if ( (address & bit_post) == 0)
			currentNode = currentNode->left;
		else
			currentNode = currentNode->right;
		bit_post = bit_post >> 1;
	}
	if (lawnode->law->protocol) free (lawnode->law->protocol);
	free (lawnode->law);

	if (lawnode->prev == NULL)
		currentNode->head = lawnode->next;
	else
		lawnode->prev->next = lawnode->next;

	if (lawnode->next == NULL)
		currentNode->tail = lawnode->prev;
	else
		lawnode->next->prev = lawnode->prev;
	free (lawnode);
	while ( (currentNode->right == NULL) &&
		(currentNode->left == NULL) &&
		(currentNode->head == NULL) &&
		(currentNode != lawtree->root) ) {
			tmp = currentNode->parent;
			if (currentNode == tmp->left)
				tmp->left = NULL;
			else
				tmp->right = NULL;

			free (currentNode);
			currentNode = tmp;
	}

	return 1;
}

/**
 * To print the list of laws associated with the TreeNode are
 * displayed on the stdout
 * 
 * @param 	node - Prints a TreeNode
 * @return 	void
 */
static void print_node (struct TreeNode *node) {
	struct LawNode *lawnode = node->head;
	if (lawnode == NULL) {
//		printf ("Empty\n");
	} 
	else {
		while (lawnode != NULL) {
		
			printf ("from %ld.%ld.%ld.%ld to %ld.%ld.%ld.%ld",
			(lawnode->law->src_addr>>24)&0xFF,(lawnode->law->src_addr>>16)&0xFF,(lawnode->law->src_addr>>8)&0xFF,lawnode->law->src_addr&0xFF,
			(lawnode->law->dst_addr>>24)&0xFF,(lawnode->law->dst_addr>>16)&0xFF,(lawnode->law->dst_addr>>8)&0xFF,lawnode->law->dst_addr&0xFF);

			printf (" ID: %d, SRC: %lX PORT: %d DST: %lX PORT: %d DIR: %d SRC_MASK: %d DST_MASK: %d PROTO: %d",lawnode->law->id, lawnode->law->src_addr, 
				lawnode->law->src_port, lawnode->law->dst_addr, lawnode->law->dst_port, lawnode->law->action, lawnode->law->src_mask, 
				lawnode->law->dst_mask, lawnode->law->protocol);
			lawnode = lawnode->next;
		}
		printf ("\n");
	}
}

/**
 * This function prints the LawTree whenever desired
 *
 * @param 	root - Prints the tree having "root" as the root of the tree
 * @return 	void
 */
static void print_tree (struct TreeNode *root) {
	if (root == NULL) return;

	print_node (root);

	if (root->left != NULL) print_tree (root->left);
	if (root->right != NULL) print_tree (root->right);
	return;
}

/**
 * The function createTree extracts the Laws from the file "firewallTree".
 * The data in the file is in the following format ->
 * 
 * "src_addr" "src_port" "dst_addr" "dst_port" "action" "src_mask" "dst_mask" "protocol"
 *
 * The above information is extracted to create a new law and it is added in the 
 * lawtree.
 * 
 * The function also keeps track of the restricted words i.e. the words that are 
 * designated as inappropriate and any network message containing any one of these
 * should be dropped. The list of these words are contained in the file "filterWords"
 * and are extracted.
 *
 * @param 	void
 * @return 	void
 */
void createTree () {
	FILE *file1 = fopen("firewallTree", "r");
	FILE *file2 = fopen("filterWords", "r");
		
	char *str;
        struct hostent *host;
	str = (char *)malloc(60);
	lawtree = (struct LawTree *)malloc(sizeof (struct LawTree));
	lawtree->root = (struct TreeNode *)malloc (sizeof (struct TreeNode));
	struct Law *newlaw;
	struct sockaddr_in dummy;
	
	if (file1) {
		newlaw = (struct Law *)malloc (sizeof (struct Law));
		while (fgets (str, 60, file1)!=NULL) {
			newlaw = (struct Law *)malloc (sizeof (struct Law));
			
			// getting the source address
			newlaw->src_addr = ntohl (inet_addr(strtok (str, " ")));
			// src port
			newlaw->src_port = atoi (strtok (NULL, " "));
			// getting the destination address
			newlaw->dst_addr = ntohl (inet_addr(strtok (NULL, " ")));
			// dst port
			newlaw->dst_port = atoi (strtok (NULL, " "));
			// getting the action of transfer
			newlaw->action = atoi (strtok (NULL, " "));
			// getting the SRC_MASK
			newlaw->src_mask = atoi (strtok (NULL, " "));
			// getting the DST_MASK
			newlaw->dst_mask = atoi (strtok (NULL, " "));
			// getting the protocol
			newlaw->protocol = atoi (strtok (NULL, " "));
			//newlaw->protocol [strlen (newlaw->protocol)-1] = '\0';
			insertLaw (newlaw);
			str = (char *)malloc(60);
		}
	}
	if (file2) {
		filter = (char **) malloc (sizeof (char *));
		int i=0;
		while (fgets (str, 10, file2) != NULL) {
			str [strlen (str)-1] = '\0';
			filter[i++] = str;
			str = (char *) malloc(10);
		}
		numWords = i;
		//for (;--i>=0;) printf ("%s\n",filter[i]);
	}
}

/**
 * The firewall_check function aims at deciding whether a message transaction 
 * from src_addr to dst_addr should be allowed or not. Various parameters are used 
 * making the decision - 
 * "src_addr" is used to find the appropriate node in the tree as the tree is source 
 * IP address built.
 * The "dst_addr", "src_port", "dst_port" and "protocol" serve as the main criteria for comparision
 *
 * The src_mask and dst_mask parameters in the rule define whether an IP address (src/dst resp.) are to 
 * be treated as mask or IP address.
 *
 * The action parameter of the rule is used to decide what the firewall needs to do with the packet being 
 * sent by the source (0 = accept and inform; 1 = drop; 2 = drop and inform)
 *
 * @param 	protocol - the protocol being followed by the packet sent by the src
 *	  	src_addr - IP address of the source in 32-bit format
 * 	  	dst_addr - IP address of the destination in 32-bit format
 *	  	src_port - The port being used by the source for communication
 * 	  	dst_port - The port being used by the destination for communication
 *
 * @return 	0 - drop packet as no law matched
 *		1 - accept packet and inform source
 *		2 - drop packet
 *		3 - drop packet and inform source
 */
static int firewall_check (int protocol, IP_addr src_addr, IP_addr dst_addr, int src_port, int dst_port) {
	IP_addr current_bit = 0x80000000; 
	struct TreeNode *cur_node = lawtree->root;
	struct LawNode *rulenode;
	struct Law *cur_rule;
	
	while (cur_node != NULL) {
		for (rulenode = cur_node->head; rulenode != NULL; rulenode = rulenode->next) {
			//printf ("Law ID = %d\n", rulenode->law->id);
			cur_rule = rulenode->law;
			if (cur_rule->dst_addr != dst_addr) { 
				//printf ("%lX != %lX\n", cur_rule->dst_addr, dst_addr); 
				if ( !(cur_rule->dst_mask == 1 && (cur_rule->dst_addr & dst_addr) == cur_rule->dst_addr) )
					continue;
				//printf ("YES\n");
			}
			//if (cur_rule->direction != direction ) continue;
			if (cur_rule->protocol != protocol) { 
			//	printf ("%d != %d\n", cur_rule->protocol, protocol); 
				continue;
			}
			if (protocol != 1 && (cur_rule->src_port != src_port || cur_rule->dst_port != dst_port)) { 
			//	printf ("%d != %d || %d != %d\n", cur_rule->src_port, src_port, cur_rule->dst_port, dst_port); 
				continue;
			}
		
			// RULE found and satisfied
			if (cur_rule->action == 0) return 1;  	// accept
			else if (cur_rule->action == 1) return 2;  	// drop
			else if (cur_rule->action == 2) return 3;  	// drop and inform
		}
	
		/* Continue to the next node according to the IPs but */
		if ( (src_addr & current_bit) == 0)
			cur_node = cur_node->left;
		else
			cur_node = cur_node->right;

		current_bit = current_bit >> 1;
	}
	return 0;
}

/**
 * The contentCheck function finds if the communication between the source 
 * destination contains any inappropriate words which have already been 
 * extracted by the firewall from the user-defined enlisted words present in 
 * the file "filterWords"
 *
 * @param	message - the message to be checked in 
 * @return	0 - inappropriate content present in message
 *		1 - no inappropriate content found
 */
int contentCheck (char *message) {
	int i = 0;
	for (i = 0; i < numWords; i++) {
		if (strstr (message, filter[i]) != NULL) return 0;
	}
	return 1;
}

/**
 * processPacket function matches the packet to its protocol
 * and also decrements the time to live (ttl) field by 1 as 
 * it is a router and should do so.
 * 
 * If the packet is TCP or UDP then it is handled appropriately by 
 * calling the suitable function to do so. In case the packet follows
 * some other protocol then it is dropped.
 *
 * It is be noted that the following firewall implementation currently handles
 * only TCP and UDP packets. Others like ICMP, ARP to name a few can be added
 * the protocols being handled as when the user desires.
 *
 * @param	buffer - the source communicates to the destination via this buffer message
 *		size   - the size of the buffer
 * @return	void
 */
void processPacket(unsigned char* buffer, int size) {
	//printf ("processing packet...\n");
	
	//Get the IP Header part of this packet
	struct ip *iphdr = (struct ip*)buffer;
	
	iphdr->ip_ttl  = iphdr->ip_ttl - 1;
	
	printf ("\nPacket received from src - %s aimed for dst - ", inet_ntoa (iphdr->ip_src));
	printf ("%s\n",inet_ntoa (iphdr->ip_dst));
	
	switch (iphdr->ip_p) //Check the Protocol and do accordingly...
	{
		case 6:  //TCP Protocol
			process_tcp_packet(buffer, size);
			break;

		case 17: //UDP Protocol
			process_udp_packet(buffer , size);
			break;

		default: //Some Other Protocol like ARP etc. then BLOCK
			printf ("\npacket blocked (unsupported protocol)\n");
			break;
	}
}

/**
 * The connection information about the source and the destination are extracted from 
 * the IP header and the TCP header. This includes the source address and port number  
 * and the same parameters for the destination. 
 *
 * A firewall_check function call is used to check if the the connection request from 
 * the source is authorized or not. In case of an authorized connection request, the 
 * contents of the message are checked against the list of blocked words.
 * 
 * Whether the source is to be informed if the router forwarded the packet to the destination
 * depends on the law parameter "action" as defined above.
 *
 * @param	buffer - the source communicates to the destination via this buffer message
 *		size   - the size of the buffer
 * @return	void
 */
void process_tcp_packet (unsigned char* buffer, int size) {
	
	struct protoTCP *tcp = (struct protoTCP*) buffer;
	IP_addr src_addr = tcp->iphdr.ip_src.s_addr;
	IP_addr dst_addr = tcp->iphdr.ip_dst.s_addr;
	// Change type of protocol in firewall_check to integer type !!!
	int protocol = tcp->iphdr.ip_p;
	
	int src_port = tcp->tcph.th_sport;
	int dst_port = tcp->tcph.th_dport;
	
	char flags = tcp->tcph.th_flags;
	
	char *message;
	int reply = 0;
	reply = firewall_check (protocol, ntohl (src_addr), ntohl (dst_addr), ntohs (src_port), ntohs (dst_port));
	
	if (reply == 1) {
		if (contentCheck ((char *) buffer + sizeof (struct protoTCP)) == 0) {
			message = "packet dropped by router (inappropriate content)\n";
			printf("%s", message);
			return;
		}
		forwardPacket (buffer, size, protocol, dst_addr, dst_port);
		message = "packet forwarded by router\n";
		printf("%s", message);
		replyBack (protocol, src_addr, src_port, message);
	}
	else {
		message = "packet dropped by router\n";
		printf("%s", message);
		if (reply == 3)	
			replyBack (protocol, src_addr, src_port, message);
		return;
	}
}

/**
 * The connection information about the source and the destination are extracted from 
 * the IP header and the UDP header. This includes the source address and port number  
 * and the same parameters for the destination. 
 *
 * A firewall_check function call is used to check if the the connection request from 
 * the source is authorized or not. In case of an authorized connection request, the 
 * contents of the message are checked against the list of blocked words.
 * 
 * Whether the source is to be informed if the router forwarded the packet to the destination
 * depends on the law parameter "action" as defined above.
 *
 * @param	buffer - the source communicates to the destination via this buffer message
 *		size   - the size of the buffer
 * @return	void
 */
void process_udp_packet (unsigned char* buffer, int size) {
	
	struct protoUDP *udp = (struct protoUDP*) buffer;
	IP_addr src_addr = udp->iphdr.ip_src.s_addr;
	IP_addr dst_addr = udp->iphdr.ip_dst.s_addr;
	int protocol = udp->iphdr.ip_p;
	
	int src_port = udp->udph.uh_sport;
	int dst_port = udp->udph.uh_dport;

	char *message;
	int reply = 0;
	reply = firewall_check (protocol, ntohl (src_addr), ntohl (dst_addr), ntohs (src_port), ntohs (dst_port));
	
	if (reply == 1) {
		if (contentCheck ((char *) buffer + sizeof (struct protoUDP)) == 0) {
			message = "packet dropped by router (inappropriate content)\n";
			printf("%s", message);
			return;
		}
		forwardPacket (buffer, size, protocol, dst_addr, dst_port);
		message = "packet forwarded by router\n";
		printf("%s", message);
		replyBack (protocol, src_addr, src_port, message);
	}
	else {
		message = "packet dropped by router\n";
		printf("%s", message);
		if (reply == 3)	
			replyBack (protocol, src_addr, src_port, message);
		return;
	}
}

/**
 * replyBack function replies back to the source with an appropriate message
 * as to whether the packet was forwarded by the router or not. Using the protocol 
 * used the message is appended after an empty packet header.
 *
 * @param 	protocol - the protocol to be used while replying back
 * 		src_addr - the IP address of the source 
 *		src_port - port being used by the source
 * 		message  - message to be forwarded
 * @return 	void
 */
void replyBack (int protocol, IP_addr src_addr, int src_port, char *message) {
	//printf ("replying back to %lX at port %d\n", ntohl (src_addr), ntohs (src_port));
	
	struct sockaddr_in router2_addr, client1_addr;
	int sock_out;
	
	if ((sock_out = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
            perror("Socket");
            exit(1);
        }

  	client1_addr.sin_family = AF_INET;     
	client1_addr.sin_port = src_port;   
	client1_addr.sin_addr.s_addr = src_addr;

	char datagram[4096];
	memset (datagram, 0, 4096);
	int size = strlen (message);
	switch (protocol) {
		case 6: strcpy ((char *) datagram + sizeof (struct protoTCP), message);
			size += sizeof (struct protoTCP);
			break;
		case 17:strcpy ((char *) datagram + sizeof (struct protoUDP), message);
			size += sizeof (struct protoUDP);
			break;
	}
	
	int sent_bytes = sendto (sock_out, datagram, size, 0, (struct sockaddr *) &client1_addr, sizeof (client1_addr));
	if (sent_bytes < 0) 
		printf ("error\n");
	//else printf ("Reply Sent bytes %d\n", sent_bytes);
	printf ("Reply message sent\n");
	
	close(sock_out);
}

/**
 * forwardPacket forwards the packet recieved from the source to the destination
 * present in the packet header. The destination IP address and port are obtained
 * as function arguements.
 *
 * @param	buffer - the buffer to be forward which contains the headers and message
 *		size   - size of the buffer including the header and message
 *		protocol - the protocol to be used while replying back
 * 		dst_addr - the IP address of the source 
 *		dst_port - port being used by the source
 * @return 	void
 */
void forwardPacket (unsigned char *buffer, int size, int protocol, IP_addr dst_addr, int dst_port) {
	struct sockaddr_in router2_addr, client2_addr;
	int sock_out;
	
	if ((sock_out = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
            perror("Socket");
            exit(1);
        }

  	client2_addr.sin_family = AF_INET;     
	client2_addr.sin_port = dst_port;   
	client2_addr.sin_addr.s_addr = dst_addr;

	//printf ("dst_port = %d ; dst_addr = %lX check = %lX\n", ntohs(dst_port), ntohl (dst_addr), ntohl (inet_addr (DST_ADDR)));
	//printf ("message = %s\n",(char *)buffer + sizeof (struct protoTCP));
	
	int sent_bytes = sendto (sock_out, buffer, size, 0, (struct sockaddr *) &client2_addr, sizeof (client2_addr));
	if (sent_bytes < 0) 
		printf ("error\n");
//	else printf ("Sent bytes = %d\n",sent_bytes);

	close(sock_out);
}

/**
 * main () begins with the creation of the LawTree that shall contain all 
 * the laws specified to the router. The router runs on a separate machine 
 * and binds itself to a port and its IP address, which are known to all the 
 * clients.
 *
 * The router then waits for any connection from a client. Any connection request 
 * to a destination is checked for authentication by the router. This is done by 
 * a call to the processPacket () call. 
 *
 * It is to be noted that the router can handle more than one clients at a time,
 * that is to say, router sequentially handles all the requests passing through it.
 *
 * @param	void
 * @return 	1
 */
int main() {
	
	createTree ();	
	//print_tree(lawtree->root);
	
        int sock_in;  
        struct sockaddr_in router_addr, client1_addr; 
        
	if ((sock_in = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
            perror("Socket");
            exit(1);
        }

	router_addr.sin_family = AF_INET;         
        router_addr.sin_port = htons(ROUTER_PORT);     
        router_addr.sin_addr.s_addr = INADDR_ANY; 
        bzero(&(router_addr.sin_zero),8); 

        if (bind(sock_in, (struct sockaddr *)&router_addr, sizeof(struct sockaddr))
                                                                       == -1) {
            perror("Unable to bind");
            exit(1);
        }
	
	fflush(stdout);

        char send_data [1024] ;
	char recv_data[1024];

	while (1) {
	
		printf("\nRouter running on IP - %s on port - %d\n", ROUTER_ADDR, ROUTER_PORT);
		memset (recv_data, 0, 4096);
		int addrlen;
		addrlen = sizeof(client1_addr);
		int recv_bytes;
		if ((recv_bytes = recvfrom(sock_in, recv_data, sizeof(recv_data) , 0, (struct sockaddr *)&client1_addr, &addrlen)) == -1) {
			perror("recv");
		}
		else {
			//printf("Received %d byte(s)\n", recv_bytes);
			processPacket(recv_data , recv_bytes);
		}
		fflush(stdout);
	}
	
      close(sock_in);
      return 0;
} 
